(ns exchange.exchange-process
  (:require [clojure.string :as s]
            [exchange.util :as u]))

; ==============================================
; 奖品兑换
; ==============================================

(comment (def status {:exchanged-num {:day {"p1" 1 "p2" 3}
                                      :all {"p1" 1 "p2" 3}}
                      :last-exchange {:username "u0"
                                      :time     0
                                      :prize    "p1"}}))

(defn mk-prizes-validator
  "返回校验奖品兑换数量的函数"
  [day-limit all-limit]
  (let [f (fn [data limit]
            (every? true? (map (fn [[k v]] (or (= v -1) (<= (data k) v))) limit)))]
    (fn [exchange]
      (and (f (:day (:exchanged-num exchange)) day-limit)
           (f (:all (:exchanged-num exchange)) all-limit)))))

(defn take-prize
  "增加兑换数"
  [exchange check-fn {:keys [prize] :as record}]
  (let [new (-> exchange
                (update-in [:exchanged-num :day prize] inc)
                (update-in [:exchanged-num :all prize] inc)
                (assoc :last-exchange record))]
    (if (check-fn new) new exchange)))

(defn reset-day-prizes
  "返回重置日兑换数的函数"
  [exchange]
  (update-in exchange [:exchanged-num :day] (partial u/map-vals (constantly 0))))

; ============================================== 
; 兑换状态数据结构
; ============================================== 

(defn mk-exchange-status
  "创建兑换状态数据结构"
  [day-limit all-limit day-exchanged all-exchanged]
  (let [init {:exchanged-num {:day day-exchanged
                              :all all-exchanged}
              :last-exchange nil}
        prizes-validator (mk-prizes-validator day-limit all-limit)
        exchange-status (atom init :validator prizes-validator)]
    {:take-prize!       (fn [{:keys [username time prize] :as record}]
                          (let [result (try (swap! exchange-status take-prize
                                                   prizes-validator record)
                                            (catch Exception e
                                              (prn "swap error" (.getMessage e))))]
                            (= record (:last-exchange result))))
     :reset-day-prizes! (fn [] (let [result (try (swap! exchange-status reset-day-prizes)
                                                 (catch Exception e
                                                   (prn "swap error" (.getMessage e))))]
                                 (map? result)))
     :exchanged-prizes  (fn [] (:exchanged-num @exchange-status))}))

(defn exchange-result
  "获得兑换的结果"
  [user-ticket prize-cost take-prize-fn]
  (cond
    (> prize-cost user-ticket) :not-enough-ticket
    (not (take-prize-fn)) :no-more-prize
    :else :ok))

; ==============================================
; 卡密信息
; ==============================================

(defn str-cards->map-cards
  "将卡密字符串解析为卡密信息"
  [card-str-lines]
  (->> card-str-lines
       (map #(when (re-matches #"^.{1,} .{1,}$" %)
              (let [s (s/split % #" ")]
                {:card (first s) :code (second s)})))
       (filter some?)))

(defn cards-enough?
  "检查卡密数量是否足够"
  [cards cards-num]
  (>= (count cards) cards-num))

(defn mk-card-code
  "返回获取一条卡密信息的函数"
  [cards exchanged-cards]
  (let [init (filter (complement (set exchanged-cards)) cards)
        remain-cards (atom [nil init])
        slurp-one (fn [[_ remain]]
                    (split-at 1 remain))]
    (fn [] (let [[a-card _] (swap! remain-cards slurp-one)]
             (first a-card)))))
