(ns hoplon.firebase.auth
  (:require [javelin.core :as j]
            [cljsjs.firebase]
            [firebase-cljs.core :as fb]
            [firebase-cljs.auth :as fbauth]
            [firebase-cljs.auth.error :as fbautherr]
            [firebase-cljs.auth.provider :as fbprov]
            [firebase-cljs.database :as fbdb]
            [firebase-cljs.user :as fbuser]
            [hoplon.firebase :as hfb])
  (:require-macros [hoplon.firebase :refer [with-auth!]]))

;; Firebase Authentication Cells ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(def ^:dynamic *user-auth*    (j/cell nil))

(def ^:dynamic *user-cred*    (j/cell nil))

(def ^:dynamic *user-fb*      (j/cell nil))

(def ^:dynamic *pending-link* (j/cell nil))

;; Firebase Authentication Providers ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(def ^:dynamic *facebook*     (fbprov/facebook))

(def ^:dynamic *github*       (fbprov/github))

(def ^:dynamic *google*       (fbprov/google))

(def ^:dynamic *twitter*      (fbprov/twitter))

(fbprov/scope-email *google* :google)
(fbprov/scope-email *github* :github)
(fbprov/scope-email *facebook* :facebook)

(fbprov/scope-profile *google* :google)
(fbprov/scope-profile *github* :github)
(fbprov/scope-profile *facebook* :facebook)

;; Firebase Authentication Helper Fn's ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn- handle-link
  [err]
  (let [cred (aget err "credential")
        email (aget err "email")
        pending {:cred cred :email email}]
    (.then
      (fbauth/providers-by-email hfb/*auth* email)
      #(reset! *pending-link*
        (assoc pending
          :providers (set %)
          :msg (fbautherr/message err))))))

(defn- handle-login
  [auth & [ref default]]
  (let [ref (when ref (fbdb/get-in ref (fbuser/uid auth)))]
    (when (nil? @*user-auth*) (reset! *user-auth* auth))
    (when (and ref default)
      (hfb/fb-default ref default))
    (when ref
      (fbdb/listen-promise ref "value" #(reset! *user-fb* (hfb/fb->clj %))))))

(defn- handle-logout
  [& [ref]]
  (dosync
   (reset! *user-auth* nil)
   (reset! *user-cred* nil)
   (when ref
     (fbdb/disable-listener! ref "value")
     (reset! *user-fb* nil))))

;; Firebase Authentication Public API ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn login!
  "Login in a user using popup based workflow.
  `provider` is required and can be any valid Firebase Provider:

        *facebook*
        *github*
        *google*
        *twitter*

  Accepts an optional `auth`, Firebase Authentication Service.

  An attribute is provided for convenience `:fb-login`, which accepts any of the
  above Firebase Providers and will cause the node's click event to call
  this function."
  [provider & [auth]]
  (-> (fbauth/login-popup (or auth hfb/*auth*) provider)
      (.then
       #(dosync
         (reset! *user-auth* (aget % "user"))
         (reset! *user-cred* (aget % "credential"))))
      (.catch
        #(case (keyword (fbautherr/code %))
          :auth/account-exists-with-different-credential (handle-link %)
          (throw (fbautherr/message %))))))

(defn logout!
  "Logout a user. Accepts an optional `auth`, Firebase Authentication Service.

  An attribute is provided for convenience `:fb-logout`, any value will cause
  the node's click event to call this function."
  [& [auth]]
  (fbauth/logout (or auth hfb/*auth*)))

(defn fb-auth!
  "An automated Firebase Authentication workflow. On successful authentication
  `*user-auth*` will be populated with the Firebase.User object.

  Optionally accepts a `ref` which can be used to locate user data in the
  Firebase Database, if `ref` exists `*user-fb*` will be populated with the
  database value. `ref` should be the parent node where users are indexed
  by uid.

  Optionally accepts a `default` user map which will be persisted to database if
  `ref` does not exist. In this case `default` will be returned and will
  populate `*user-fb*`.

  On logout `*user-auth*` and `*user-fb*` are set to `nil`."
  [& [ref default]]
  (with-auth! #(handle-login % ref default) #(handle-logout ref)))

;; Firebase Authentication Attributes ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmethod hoplon.core/do! :fb-login
  [elem _ v]
  (.addEventListener elem "click" #(login! v)))

(defmethod hoplon.core/do! :fb-logout
  [elem _ v]
  (.addEventListener elem "click" #(logout!)))
