(ns hub.mailer.client
  "Mailer client for RaceHub"
  (:require  [com.stuartsierra.component :as c]
             [hub.mailer.service :as m]
             [hub.mailer.service.user :as su]
             [hub.mailer.service.registration :as sr]
             [hub.queue.client :as qc]
             [hub.queue.schema :as qs]
             [schema.core :as s])
  (:import [com.stuartsierra.component Lifecycle]))

;; ## API

;; might want to make these private, since the intention is that they
;; are called indirectly via the queue.

(s/defn user-created
  [{:keys [type username email profile] :as m} :- qs/UserCreated]
  (when (= type "full-user")
    (println "Mailer: User Created.")
    (su/send-welcome-email! {:username username
                             :first-name (get-in profile [:name :first] "there")
                             :to-email (:address email)})))

(s/defn user-updated
  "TODO"
  [m]
  )

(s/defn trigger-password-reset
  [{:keys [email] :as m} :- qs/TriggerPasswordReset]
  (println "Mailer: Password Reset.")
  (su/email-pw-reset-code! {:to-email email
                            :first-name (get-in m [:name :first] "there")
                            :reset-code (get-in m [:password-reset :code])}))

(s/defn reg-created
  "TODO"
  [m])

(s/defn reg-updated
  "TODO"
  [m])

;; ## Setup
(defonce system (atom nil))

(def default-topic-fns
  "Map of queue topic to mailer API handler function (of the event)."
  {"user-created" user-created
   "user-updated" user-updated
   "reg-created" reg-created
   "reg-updated" reg-updated
   "trigger-password-reset" trigger-password-reset})

(defrecord MailerClient [topic->fn]
  c/Lifecycle
  (start [this]
    (if (:unsub-fns this)
      this ;;already started
      (let [unsubs (doall (map (fn [[topic handler]] (qc/subscribe! topic handler)) topic->fn))]
        ;; return unsubscribe functions
        (assoc this :unsub-fns unsubs))))
  (stop [{:keys [queue unsub-fns] :as this}]
    (when unsub-fns
      (doall (map (fn [unsub] (unsub)) unsub-fns)))
    this))

(s/defn mailer-client
  "Creates a lifecycled mailer client. You can specify a map of topic
  to handler function. The fn should have one arg - the topic payload
  that comes in via queue.

  The queue client and the channels it uses are hidden - just returns
  a reified object that has functions to start and stop the mailer
  component. Starting will create a queue client, and subscribe to
  specified topics, dispatching the the appropriate functions. By
  default, topic messages are dispatched to the mailer API functions
  above."
  [{:keys [spec topic-fns]
    :or {spec {}
         topic-fns default-topic-fns}}
   :- {(s/optional-key :spec) qc/RedisSpec
       (s/optional-key :topic-fns) {s/Str (s/=> s/Any s/Any)}}]
  (c/system-map
   :queue (qc/queue-client spec)
   :mailer (c/using (->MailerClient topic-fns) [:queue])))

(defn startup!
  "Starts a mailer-client using the default topic->function mapping."
  []
  (reset! system (mailer-client {}))
  (c/start @system))

(defn stop!
  "Stops the mailer client."
  []
  (c/stop @system)
  (reset! system nil))
