(ns hub.mailer.service
  "Implementation code."
  (:require [clojure.core.async :as a]
            [clojure.java.io :as io]
            [hub.mailer.conf :as conf]
            [mandrill.core :as mandrill]
            [net.cgrand.enlive-html :as h]
            [schema.core :as s])
  (:import [java.io File]))

;; ## Schemas

(def TemplateName
  (s/named s/Str "Enlive email Template name"))

;;TODO: Feedback should come in via the queue.
(def Feedback
  {:local_time s/Str
   :user_agent s/Str
   :location (s/named s/Str "URL the user was at when they sent feedback.")
   :content s/Str
   :email s/Str
   (s/optional-key :username) s/Str
   (s/optional-key :full-name) s/Str})

(s/defschema EMail
  "Incoming request to send an email using our Mandrill account."
  {:to-email (s/either mandrill/Address [mandrill/Address])
   :subject s/Str
   :body (s/named s/Any "Enlive body.")
   (s/optional-key :from-email) mandrill/Address
   (s/optional-key :unsubscribe-link?) s/Bool
   s/Any s/Any})

(s/defschema Function
  (s/=> s/Any s/Any))

;; ## Templates

(def email-path "resources/email")

(s/defn template :- File
  "Returns the path to the html template with the supplied name."
  [template-name :- TemplateName]
  (let [file (io/file email-path (str template-name ".html"))]
    (assert (.exists file) (format "Template %s doesn't exist!" template-name))
    file))

;; ## Global Template
(s/defn replace-vars :- Function
  "Wraps h/replace-vars. Makes sure all global RaceHub config vars are
  replaced for mailer templates."
  [m :- {s/Any s/Any}]
  (h/replace-vars
   (assoc m :current-server (conf/current-server)
          :site-name (conf/site-name))))

(defn- external-url? [^String url]
  (re-matches #"^(?://|http://|https://).*" url))

;;TODO:
;; Img source URIs, notification URIs, all hrefs. Need coordination
;; with RH.

;; Problem: Template in mailer services relies on imgs being served
;; from racehub web app. Need to coordinate URIs.

;; Solution:

;; Right now the ONLY image is the logo. Lets put it in an S3 bucket
;; and use that url.

(h/deftemplate shell
  (template "responsive_shell")
  [body & [subject unsubscribe?]]
  [:title] (if subject
             (h/content subject)
             identity)
  [:a#img-link] (h/set-attr :href (conf/current-server))
  [:td.email] (h/content body)
  [:p.unsubscribe] (when unsubscribe?
                     #(h/at % [:a] (h/set-attr :href
                                               ;;TODO: Fix this!
                                               (format "%s/profile/notifications"
                                                       (conf/current-server))))))

(s/defn send-email :- (s/maybe (s/named s/Any "Core Async Channel"))
  "Sends an email via Mandrill using RaceHub's responsive template"
  [{:keys [body subject unsubscribe-link?] :as m} :- EMail]
  (-> (dissoc m :from-email)
      (assoc :from {:email (:from-email m
                                        (if (= (conf/site) :pg)
                                          "support@paddleguru.com"
                                          "support@racehubhq.com"))
                    :name (conf/site-name)}
             :body (shell body subject unsubscribe-link?))
      (dissoc :unsubscribe-link?)
      (mandrill/send-message {:out-ch (a/chan)})))
