(ns hub.mailer.service.registration
  "Mailer for registration related events (created, updated)."
  #_(:require [clojure.string :as st]
              [net.cgrand.enlive-html :as h]
              [racehub.api.register.schema :as rs]
              [racehub.config :as conf]
              [racehub.couchdb :as db]
              [racehub.models.event-entry :as ee]
              [racehub.models.event :as event]
              [racehub.models.extras :as extras]
              [racehub.models.race :as race]
              [racehub.models.user :as user]
              [racehub.route :as route]
              [racehub.schema :as ps]
              [racehub.util :as u]
              [racehub.util.mail :as mail]
              [racehub.util.log :as log]
              [schema.core :as s])
  (:import [java.io File]))

;; ## Schema

;; TODO: Think about what schema we actually want here, since all
;; these things live in different services.

#_(do
    (s/defschema EmailInfo
      "Info needed for user confirmation emails and admin notification
  emails for both new and udpated registrations."
      {:payment-type race/PaymentType
       :orders {extras/ItemID extras/Order}
       :ee-partitions ee/EntryPartitions
       :event-docs [event/Event]
       :extra-items [extras/ExtraItem]
       :registration-type rs/RegistrationType
       :user user/UserDoc
       :race-doc race/Regatta
       :reg-url (s/named s/Str "URL at which the user (or admin) can edit the registration.")
       :balance (s/named ps/Currency "Amount charged on the current submission (receipt).")
       :total-paid (s/named ps/Currency "Total paid (cumulatively) for the registration.")})

    ;; Helpers

    (s/defn build-userid->info :- user/UserID->Info
      "Takes in a map of event entries, and builds out a map of
  user-id (from the entry athletes) to their username and full-name."
      [ee-partitions :- ee/EntryPartitions]
      (->> (vals ee-partitions)
           (apply concat)
           (ee/get-athletes)
           (user/get-info-for-ids)))

    ;; ## Snippets

    ;; Info for one given entry, for an event.
    (h/defsnippet entry-info
      (mail/template "race/entry_info")
      [:div.entries]
      [{:keys [craft-type] :as replacements}]
      [[:li h/last-of-type]] (if craft-type
                               (h/after (h/html [:li (str "Craft Type: " craft-type)]))
                               identity)
      [h/any-node] (mail/replace-vars (update replacements :age-group :name)))

    ;;This template handles new user registrations as well as user edits
    (h/defsnippet registration-confirmation-email
      (mail/template "race/registration")
      [:body :> h/any-node]
      [{:keys [ee-partitions userid->info reg-url note event-docs orders extra-items balance
               payment-type race-doc userid->info user total-paid registration-type
               admin-email?]}]
      [:span.intro] (h/substitute
                     (let [edit? (:update? registration-type)]
                       (if admin-email?
                         (str (user/display-name user) " has just "
                              (if edit?
                                "edited"
                                "submitted")
                              " their registration for ")
                         (if edit?
                           "You have successfully updated your registration for"
                           "This email confirms your registration for"))))
      [:span.confirm] (h/substitute
                       (h/html
                        (if admin-email?
                          [:span "You are receiving this email in accordance with your notification settings, which you can edit by clicking the link at the bottom of this email."]
                          [:span [:a {:href (str (conf/get-config :current-server) reg-url)}
                                  "Click here"]
                           " if you would like to make any edits to this registration. You can also track all of your past and upcoming races in your RaceHub profile."])))
      [:p.entries] (if (seq ee-partitions)
                     identity
                     (h/substitute nil))
      [:div.entries] (h/clone-for
                      [entry (concat (:update ee-partitions) (:create ee-partitions))
                       :let [event-name (some (fn [event]
                                                (when (= (:_id event) (:event-id entry))
                                                  (:name event)))
                                              event-docs)
                             athletes (->> (concat [(db/get-id user)]
                                                   (when-let [partners
                                                              (not-empty (:partners entry))]
                                                     partners))
                                           (map (fn [uid]
                                                  (:full-name (userid->info uid) "")))
                                           (st/join ", "))]]
                      (h/substitute (entry-info (merge (:division entry)
                                                       {:athletes athletes
                                                        :event-name event-name}))))
      [:p.orders] (if (seq orders)
                    identity
                    (h/substitute nil))
      [:div.order] (h/clone-for [[item-id order] orders]
                                (h/content
                                 (h/html
                                  [:p (format "%s: %s"
                                              (:label (extras/item-for-order order extra-items))
                                              (u/pennies->currency (:price order)))])))
      [:div.fee] (let [balance (u/pennies->currency balance)
                       total-paid (u/pennies->currency total-paid)]
                   (h/content
                    (h/html [:p (str "Total Fee: " total-paid)]
                            (if (:update? registration-type)
                              [:p (str "Amount From This Edit: " balance)]
                              [:p ""]))))
      [:p.note] (if (seq note)
                  identity
                  (h/substitute nil))
      [:div.note] (if (seq note)
                    (h/do-> (h/content (h/html-snippet note))
                            (h/append (h/html [:br] [:br])))
                    (h/substitute nil))
      [h/any-node] (mail/replace-vars {:title (:title race-doc "")
                                       :full-name (if admin-email?
                                                    "there"
                                                    (user/display-name user))
                                       :payment-note (case payment-type
                                                       "credit-card" "Credit Card (online)"
                                                       "pledge" "Pledge (will pay on race day.)"
                                                       "offline" "Offline Payment"
                                                       payment-type)}))

    ;; ## Sending Mail

    (s/defn registration-confirmation :- ps/Channel
      "Sends out a registration confirmation email for a new user
  registration, or a user editing their registration."
      [ring-req :- route/RingReq
       {:keys [email race-doc ee-partitions user] :as email-info} :- EmailInfo]
      (let [extra {:note (get-in race-doc [:registration :email-confirmation-content])
                   :userid->info (build-userid->info ee-partitions)}]
        (mail/send-email ring-req
                         {:to-email (user/email user)
                          :subject (str "RaceHub Registration: " (:title race-doc))
                          :body (registration-confirmation-email (merge email-info extra))})))

    (s/defn registration-admin-notification :- ps/Channel
      "Sends out a registration notification email to the admins
  subscribed to new or edited registrations for their race."
      [ring-req :- route/RingReq
       {:keys [email race-doc ee-partitions user registration-type] :as email-info} :- EmailInfo
       admin-emails :- [user/EMail]]
      (mail/send-email ring-req
                       {:to-email admin-emails
                        :subject (str (if (:update? registration-type)
                                        "Edited Registration: "
                                        "New Registration: ")
                                      (:title race-doc))
                        :body (registration-confirmation-email
                               (assoc email-info :userid->info (build-userid->info ee-partitions)
                                      :admin-email? true))
                        :unsubscribe-link? true})))
