(ns hub.photo.client
  (:require [clj-time.coerce :refer [to-long]]
            [clj-time.core :as ct]
            [schema.core :as s]
            [hub.photo.schema :as ps]
            [hub.photo.service :as photo]
            [hub.photo.setup :refer [setup!]]
            [hub.util.api :as ua]
            [hub.util.rethink :as ur :refer [RethinkSpec]]
            [rethinkdb.query :as r]
            [rethinkdb.query-builder :as qb])
  (:import [com.stuartsierra.component Lifecycle]))

;; # API

(s/defschema PhotoInput
  {:url ps/PhotoURL
   (s/optional-key :location) ps/GeoTag
   (s/optional-key :dimensions) ps/Dims
   (s/optional-key :timestamp) s/Int})

;; ## Create

(s/defn create! :- [ps/ID]
  "Generates a bunch of photo instances for the supplied race off of
  the supplied photo URLs."
  [race-id :- ps/RaceID
   photos :- [PhotoInput]]
  (:generated_keys
   (photo/run-photo
    (r/insert
     (map (fn [{:keys [url location timestamp dimensions]}]
            (merge
             {:race-id race-id
              :photo-url url
              :tags []
              :timestamp (or timestamp (to-long (ct/now)))}
             (when location {:location location})
             (when dimensions {:dimensions dimensions})))
          photos)))))

;; ## Non-Photo Updates

(s/defn set-privacy! :- s/Bool
  "Registers a default privacy setting for the specified user. All new
  photos tagged will show up with this privacy setting across ALL
  races."
  [m :- {:user-id ps/UserID, :privacy-level ps/Privacy}]
  (try (let [table (r/table photo/privacy-table)
             result (photo/run
                      (r/table photo/privacy-table)
                      (r/get-all [(:user-id m)] {:index "by-user-id"})
                      (r/is-empty)
                      (r/branch
                       (r/insert table m)
                       (r/get-all table [(:user-id m)] {:index "by-user-id"})))]
         true)
       (catch Exception _ false)))

(s/defn register-bib-mapping! :- s/Bool
  "For the supplied race ID, registers a mapping of bib number to user
  ID. Optionally set a default privacy for the user; if the user has a
  custom default privacy setting registered, the one supplied here
  will have no effect.

  This call will potentially trigger user tag events on the event
  stream (if there are photos with tagged bibs and no linked photos,
  for example)."
  [race-id :- ps/RaceID
   m :- {ps/Bib {:user-id ps/UserID
                 (s/optional-key :default-privacy) ps/Privacy}}]
  (try (if-let [existing (photo/get-bib-mapping race-id)]
         (photo/run
           (r/table photo/bib-mapping)
           (r/replace (assoc existing :mapping m)))
         (photo/run
           (r/table photo/bib-mapping)
           (r/insert {:race-id race-id
                      :mapping m})))
       true
       (catch Exception _ false)))

;; ## Get

(s/defn multiget :- {ps/ID ps/Photo}
  "Multiget, same as it always works."
  [ids :- [ps/ID]]
  (photo/multiget ids))

(def get-photo
  (photo/multiget->get multiget))

(s/defn photos-by-races :- {ps/RaceID [ps/Photo]}
  "Accepts a race ID and returns a sequence of photos for that race."
  [race-ids :- [ps/RaceID]]
  (if-let [ids (not-empty (distinct race-ids))]
    (->> (photo/run-photo
          (r/get-all ids {:index "by-race-id"}))
         (photo/hydrate-users)
         (group-by :race-id))
    {}))

(s/defn photos-by-bib :- [ps/Photo]
  "Accepts a race ID and a bib number and returns a sequence of photos
  for that bib number at that race."
  [race-id :- ps/RaceID bib-id :- ps/Bib]
  (photo/run-photo
   (r/get-all [[race-id bib-id]] {:index "by-bib-id"})))

(s/defn photos-by-user :- {ps/RaceID [ps/Photo]}
  "Accepts a set of user IDs and returns a map of user id to a map of
  RaceID to a sequence of photos that the user appears in (alongside
  the privacy level)."
  [race-ids :- [ps/RaceID] user-id :- ps/UserID]
  (->> (photos-by-races race-ids)
       (mapcat (fn [[race-id photos]]
                 (let [filtered (filter
                                 (fn [photo]
                                   (some (comp #{user-id} :user-id :user-tag)
                                         (:tags photo)))
                                 photos)]
                   (if (not-empty filtered)
                     [[race-id filtered]]))))
       (into {})))

;; Photo Updates

(letfn [(if-photo-exists [id f]
          (if-let [photo (photo/run-photo (r/get id))]
            (f photo)
            {:error
             {:type "photo_doesnt_exist"
              :message (format "Photo with id %s doesn't exist." id)}}))]

  (s/defn geotag! :- (s/either ps/Photo ua/Err)
    "Applies the supplied geotag to the supplied photo."
    [id :- ps/ID geotag :- ps/GeoTag]
    (if-photo-exists
     id
     (fn [photo]
       (let [updated (assoc photo :location geotag)]
         (photo/run-photo
          (r/replace updated))
         updated))))

  (s/defn tag! :- (s/either ps/Photo ua/Err)
    "Marks a tag for the supplied photo ID. If you supply one of
    x-offset and y-offset, the default is zero."
    [photo-id :- ps/ID
     tags :- [{:bib ps/Bib
               (s/optional-key :x-offset) s/Num
               (s/optional-key :y-offset) s/Num}]]
    (if-photo-exists
     photo-id
     (fn [photo]
       (let [tags (map (fn [{:keys [bib x-offset y-offset]}]
                         (merge {:bib bib}
                                (when (or x-offset y-offset)
                                  {:coords {:x-offset (or x-offset 0)
                                            :y-offset (or y-offset 0)}})))
                       tags)
             updated (assoc photo :tags tags)]
         (photo/run-photo
          (r/replace updated))
         updated)))))

(s/defn merge-users! :- (s/either s/Bool ua/Err)
  "Merges the sub-id user into the primary-id user. Returns the merged
  user. Errors if either user doesn't exist.

  This can be used to merge full users, OR to merge pending users
  together or into full users. Can't merge full users into a pending
  user."
  [primary-id :- ps/UserID
   sub-id :- ps/UserID]
  (let [affected-mappings (photo/run
                            (r/table photo/bib-mapping)
                            (r/get-all [sub-id] {:index "by-user-id"}))
        user-id->privacy (photo/multiget-privacy-info [primary-id sub-id])
        updated (for [m affected-mappings]
                  (update m :mapping
                          (fn [mapping]
                            (photo/map-values
                             (fn [{:keys [user-id] :as user-m}]
                               (if (= user-id sub-id)
                                 (assoc user-m :user-id primary-id)
                                 user-m))
                             mapping))))]
    (try (let [pri (user-id->privacy primary-id)
               sub (user-id->privacy sub-id)]

           (cond (and pri sub)
                 (photo/run
                   (r/table photo/privacy-table)
                   (r/delete sub))
                 pri nil
                 sub (photo/run
                       (r/table photo/privacy-table)
                       (r/replace (assoc sub :user-id primary-id)))))
         (photo/run
           (r/table photo/bib-mapping)
           (r/insert updated {:conflict "replace"}))
         true
         (catch Exception _ false))))

;; ## API Initialization

(s/defn photo-client :- Lifecycle
  "Client for the photo service."
  ([]
   (photo-client {:rethink {:mode :live
                            :db "racehub"
                            :spec {}}}))
  ([opts :- {:rethink {:mode (s/enum :test :live)
                       :db s/Str
                       :spec RethinkSpec}}]
   (let [{:keys [mode db spec]} (:rethink opts)]
     (ur/rethinkdb mode db spec setup!))))
