(ns hub.photo.schema
  "Photos service schema."
  (:require [schema.core :as s]))

;; # Schema

(def ID s/Str)

(s/defschema GeoTag
  {:lat s/Num
   :lng s/Num})

(s/defschema PixelCoords
  {:x-offset s/Int
   :y-offset s/Int})

(s/defschema Privacy
  (s/enum "dont-tag" "private" "friends" "public"))

(def UserID s/Str)

(s/defschema UserTag
  {:user-id UserID
   :privacy-level Privacy
   (s/optional-key :facebook-id) s/Str})

(def Bib s/Str)

(s/defschema Tag
  {:bib Bib
   (s/optional-key :coords) PixelCoords
   (s/optional-key :user-tag) UserTag})

(s/defschema Dims
  "height and width in pixels."
  {:height s/Int
   :width s/Int})

(def PhotoURL
  (s/named s/Str "Location of the photo. This should be a relative URL
  of the location in our S3 bucket."))

(def RaceID s/Str)
(def AlbumID
  "Facebook Album ID. Present if photo's been uploaded to Facebook."
  s/Str)

(s/defschema Source
  "Photo attribution details."
  {:description s/Str
   (s/optional-key :url) s/Str})

(def FacebookID
  (s/named s/Str "ID of the photo on Facebook."))

(s/defschema Photo
  "Schema for the photo object."
  {:id ID
   :race-id RaceID
   :photo-url PhotoURL
   :tags [Tag]
   :timestamp s/Int
   (s/optional-key :facebook-id) FacebookID
   (s/optional-key :album-id) AlbumID
   (s/optional-key :source) Source
   (s/optional-key :location) GeoTag
   (s/optional-key :filename) s/Str
   (s/optional-key :dimensions) Dims})

(s/defschema PhotoInput
  {:url PhotoURL
   (s/optional-key :location) GeoTag
   (s/optional-key :filename) s/Str
   (s/optional-key :dimensions) Dims
   (s/optional-key :timestamp) s/Int})

;; Privacy Table

(s/defschema UserTagDoc
  "Info stored in the user information table, still called the
  privacy-table."
  {:id s/Str
   :user-id UserID
   (s/optional-key :privacy-level) Privacy
   (s/optional-key :facebook-id) s/Str})

;; Race Info

(s/defschema RaceInfo
  {:id ID
   :race-id RaceID
   :mapping {Bib {:user-id UserID
                  ;; Privacy setting for THIS RACE'S photos.
                  (s/optional-key :default-privacy) Privacy}}
   (s/optional-key :album-id) AlbumID})
