(ns hub.photo.facebook
  (:require [cheshire.core :refer [parse-string generate-string]]
            [hub.photo.config :as conf]
            [hub.photo.schema :as ps]
            [org.httpkit.client :as http]
            [hub.util.facebook :as fb]
            [schema.core :as s]
            [taoensso.timbre :as timbre])
  (:import [java.util UUID]))

;; ## Schema
(s/defn make-required :- {s/Any s/Any}
  "Takes in a Schema, and a keyword to make required."
  [schema :- {s/Any s/Any}
   k :- (s/named s/Any "Key to toggle")]
  (-> (assoc schema k (get schema (s/optional-key k)))
      (dissoc (s/optional-key k))))

(s/defschema LinkedPhoto
  "A Photo that has been uploaded to FB."
  (-> ps/Photo
      (make-required :facebook-id)
      (make-required :album-id)))

;; ## API

(def query-string
  @#'http/query-string)

;; ### Photo Uploading

(s/defn ^:private link-photos-dev :- [LinkedPhoto]
  [album-id :- ps/AlbumID
   cdn-prefix :- s/Str
   photo-id->caption :- {ps/ID s/Str}
   inputs :- [ps/Photo]]
  (map #(assoc %
               :facebook-id (str (UUID/randomUUID))
               :album-id album-id)
       inputs))

(s/defn ^:private link-photos-prod* :- [{:id s/Str :post_id s/Str}]
  "One batch of FB photo upload requests. Returns FB responses - one
  per photo in the batch."
  [album-id :- ps/AlbumID
   cdn-prefix :- s/Str
   photo-id->caption :- {ps/ID s/Str}
   inputs :- [ps/Photo]]
  ;;https://developers.facebook.com/docs/graph-api/making-multiple-requests
  (let [{:keys [page-id page-admin-token]} (conf/facebook-config)
        relative-url (fn [opts]
                       (str (str album-id "/photos") "?"
                            (query-string opts)))
        opts {:query-params
              {:access_token page-admin-token
               :batch (generate-string
                       (for [input inputs]
                         {:method "POST"
                          :relative_url
                          (relative-url {:url (str cdn-prefix (:photo-url input))
                                         :caption (get photo-id->caption (:id input))})}))}}]
    (map (fn [{:keys [body]}]
           (let [result (parse-string body keyword)]
             (timbre/info {:service "photo"
                           :api-call "fb-photo-upload"
                           :album-id album-id
                           :facebook-response result})
             result))
         (fb/body
          (http/post "https://graph.facebook.com" opts)))))

(s/defn ^:private link-photos-prod :- [LinkedPhoto]
  "Uploads photos to FB in batches of up to 40 photos."
  [album-id :- ps/AlbumID
   cdn-prefix :- s/Str
   photo-id->caption :- {ps/ID s/Str}
   inputs :- [ps/Photo]]
  (timbre/info {:service "photo"
                :api-call "link-photos-prod"
                :args {:album-id album-id
                       :cdn-prefix cdn-prefix
                       :photo-id->caption photo-id->caption
                       :photo-count (count inputs)}})
  (let [batches (partition-all 40 inputs)
        fb-result (if (> (count batches) 1)
                    (->> batches
                         (map-indexed
                          (fn [idx batch]
                            (let [res (link-photos-prod* album-id cdn-prefix
                                                         photo-id->caption batch)]
                              (timbre/info {:service "photo"
                                            :api-call "fb-photo-upload"
                                            :album-id album-id
                                            :batch-number (inc idx)
                                            :message "Batch complete, sleeping!"})
                              (Thread/sleep 5000)
                              res)))
                         (apply concat))
                    (link-photos-prod* album-id cdn-prefix photo-id->caption inputs))]
    (map (fn [m {fb-id :id}]
           (assoc m :facebook-id fb-id, :album-id album-id))
         inputs fb-result)))

(s/defn link-photos! :- [LinkedPhoto]
  "Uploads the supplied photos to Facebook and adds the facebook id
  and album id into the photo input. These photos can then be added to
  the database."
  [album-id :- ps/AlbumID
   cdn-prefix :- s/Str
   photo-id->caption :- {ps/ID s/Str}
   inputs :- [ps/Photo]]
  (let [link (if (conf/prod?)
               link-photos-prod
               link-photos-dev)]
    (link album-id cdn-prefix photo-id->caption inputs)))

(s/defn get-photo
  "retrieve photo info from Facebook. See
  https://developers.facebook.com/docs/graph-api/reference/photo/ for
  return schema."
  [photo-id :- ps/FacebookID]
  (fb/api-get photo-id
              {:params
               {:oauth-token
                (fb/server-secret)}
               :query-params {}}))

;; ### Tagging

(s/defn tag-photos-prod :- [{:success s/Bool}]
  "Tags all photos on Facebook. Returns a sequence of tag responses."
  [tags :- [{:photo-fb-id ps/FacebookID
             :user-fb-ids [(s/maybe s/Str)]}]]
  (let [opts {:query-params
              {:access_token (:page-admin-token
                              (conf/facebook-config))
               :batch (generate-string
                       (for [{:keys [photo-fb-id user-fb-ids]} tags
                             id user-fb-ids :when id]
                         {:method "POST"
                          :relative_url (str photo-fb-id "/tags?tag_uid=" id)}))}}]
    (map (comp #(parse-string % keyword) :body)
         (fb/body
          (http/post "https://graph.facebook.com" opts)))))

;; ### Albums

(s/defn create-album! :- {:id s/Str}
  "Creates a new facebook album and returns the ID of that album. In
  dev mode, just returns a fake UUID.

  Reference: https://developers.facebook.com/docs/graph-api/reference/page/albums"
  [m :- {:name s/Str
         :location s/Str}]
  (if (conf/prod?)
    (let [{:keys [page-id page-admin-token]} (conf/facebook-config)]
      (fb/api-post
       (str page-id "/albums")
       {:params {:oauth-token (fb/server-secret)}
        :query-params (assoc m :access_token page-admin-token)}))
    {:id (str (UUID/randomUUID))}))

(s/defn album-url :- s/Str
  "Returns the URL for the supplied album ID."
  [album-id :- s/Str]
  (:link (fb/api-get
          album-id
          {:query-params {:access_token
                          (:page-admin-token
                           (conf/facebook-config))}})))
