(ns hub.user.service
  "Service implementation details."
  (:require [schema.core :as s]))

;; We need a table to get a user by the ID. It should ALWAYS return
;; the actual user. This way we only ever have one link in the
;; chain. Easy with RethinkDB and joins.

;; ## API:
;;
;; Create user
;; Get user (by various types)
;; Updates - oauth changes, notification, profile, etc
;; merge users (used to convert pending users to real user and merge real.)

(defn setup!
  "Basic setup for the user table."
  []
  (with-open [c (connect)]
    (-> (sr/try-create "racehub")
        (r/run c))

    (-> (sr/try-table "racehub" "user")
        (r/run c))

    (-> (r/branch
         (-> (r/index-list
              (-> (r/db "racehub")
                  (r/table "user")))
             (r/contains "username"))
         true
         (-> (r/db "racehub")
             (r/table "user")
             (r/index-create "username"
                             (r/fn [row]
                               (r/downcase
                                (r/get-field row :username))))))
        (r/run c))))

;; ## Create

(s/defn username-valid? :- s/Bool
    [s :- s/Str]
    (boolean
     (re-matches #"[a-zA-Z0-9]{3,20}" s)))

(s/defschema GenOptions
  "Options for username generation."
  {:email s/Str
   (s/optional-key :first-name) s/Str
   (s/optional-key :last-name) s/Str})

(s/defn random-usernames
  "Don't put an output schema on this! Returns an infinite sequence
    of usernames created by sticking the prefix onto the beginning of
    random numbers below the supplied maximum value."
  [prefix :- s/Str
   max :- s/Int]
  (map (partial str prefix) (repeatedly #(rand-int max))))

(s/defn get-many-users-map :- {s/Str User}
  "Takes in a collection of usernames, and returns a map of username
  to user doc for all of them."
  [names :- [s/Str]]
  (with-open [c (connect)]
    (into {} (for [d (-> (r/db "racehub")
                         (r/table "user")
                         (r/get-all (set names) {:index "username"})
                         (r/run c))]
               [(:username d) d]))))

(s/defn winner :- (s/maybe s/Str)
  "Queries the database for all supplied username
    candidates. Returns the first candidate that doesn't already exist
    in the database, or nil if they all do."
  [candidates :- [(s/maybe s/Str)]]
  (let [candidates (->> candidates
                        (remove (some-fn empty? (complement username-valid?)))
                        (map str/lower-case))
        fetched (get-many-users-map candidates)]
    (some (fn [k] (when-not (fetched k) k)) candidates)))

(s/defn generate-username :- s/Str
  "Generates a random username for the supplied GenOptions map of
    seeding options. Tries concatenating the first and last names,
    taking just the first initial and last name, and the email
    prefix (before the @ sign). After that, attempts to generate a
    username.  NOT guaranteed to terminate, but extremely likely :)"
  [{:keys [email first-name last-name]} :- GenOptions]
  (let [email-prefix            (first (str/split email #"@"))
        first-initial-last-name (str (first first-name) last-name)
        first-tries [email-prefix
                     first-initial-last-name
                     (str first-name last-name)]
        un-prefix (or email-prefix last-name)]
    (loop [candidates first-tries seed 1000]
      (or (winner candidates)
          (recur (take 10 (random-usernames un-prefix seed))
                 (* seed 10))))))

(defmulti generate-user
  "Generates a user document off of the information provided for the
  supplied type. Signup gives username and password only, for
  example. Facebook gives a bit more."
  (fn [type data] type))


(defn encrypt
  "Encrypts a string value using scrypt.
   Arguments are:
   raw (string): a string to encrypt
   :n (integer): CPU cost parameter (default is 16384)
   :r (integer): RAM cost parameter (default is 8)
   :p (integer): parallelism parameter (default is 1)
   The output of SCryptUtil.scrypt is a string in the modified MCF format:
   $s0$params$salt$key
   s0     - version 0 of the format with 128-bit salt and 256-bit derived key
   params - 32-bit hex integer containing log2(N) (16 bits), r (8 bits), and p (8 bits)
   salt   - base64-encoded salt
   key    - base64-encoded derived key"
  [raw & {:keys [n r p]
          :or {n 16384 r 8 p 1}}]
  (sc/encrypt raw n r p))

(s/defmethod generate-user :signup
  [_ {:keys [email password]} :- {:email s/Str, :password s/Str}]
  {:password (encrypt password)
   :username (generate-username {:email email})
   :email {:address email, :verified? false}
   :profile {}})

(s/defmethod generate-user :default
  [_ m :- {s/Any s/Any}]
  (cond-> m
    (:username m) (update :username str/lower-case)
    (-> m :email :address) (update-in [:email :address] str/lower-case)
    (:password m) (update :password encrypt)))

(s/defschema SignupType
  (s/enum :signup :facebook :default))

(s/defn create! :- User
  "Creates a user document using the supplied method."
  ([data] (create! :default data))
  ([type :- SignupType data]
   (with-open [c (connect)]
     (-> (r/db "racehub")
         (r/table "user")
         (r/insert [(generate-user type data)])
         (r/run c)))))
