(ns hub.user.client
  "User Service client."
  (:require [clojure.string :as str]
            [com.stuartsierra.component :as c]
            [finagle-clojure.futures :as f]
            [hub.user.api.schema :as as]
            [hub.user.service :as u]
            [hub.user.transforms :as t]
            [schema.core :as s]))

;; # User Client

;; ## Helpers
  (s/defn profile :- (s/maybe as/Profile)
    "If the supplied user's valid, returns the profile."
    [user-doc :- as/User]
    (:profile user-doc))

  (s/defn email :- (s/maybe s/Str)
    "If the supplied user's valid, returns the email address."
    [user-doc :- as/User]
    (-> user-doc :email :address))

  (s/defn display-name :- (s/maybe (s/either as/FullName as/UserName))
    "Takes in a user-doc or username and returns the user's full name,
    or username if the name hasn't been input yet."
    [user-doc :- as/User]
    (if-let [{:keys [first last]} (-> user-doc profile :name)]
      (format "%s %s" first last)
      (:username user-doc)))

  (s/defn get-dob-str :- (s/maybe s/Str)
    "Returns date of birth as a string mm/dd/yyyy"
    [user-doc :- as/User]
    (when-let [birthdate (-> user-doc :profile :birthdate)]
      (->> (map birthdate [:month :day :year])
           (str/join "/"))))

;; ## Create

(s/defn create! :- (s/either as/User as/Err)
  "Generates a user off of the supplied email and password."
  [m :- {:email s/Str :password s/Str}]
  (t/user-model->user-api (u/create! :signup m)))

(s/defn create-pending! :- (s/either as/User as/Err)
  "Generates a pending user instance. Returns an error if a user
  exists with the supplied email."
  [m :- as/PendingUserInput])

;; ## Facebook
;;
;; These are pretty easy to extend to the other oauth providers.

(s/defn register-facebook! :- (s/either as/User as/Err)
  "Returns a user off of the supplied facebook information. Optionally
   takes a UserID to associate the FaceBook info with.

  - If a user already exists with that facebook id, upgrades their
    token.

  - errors if another user is using the facebook account's email and
    ALREADY has another linked facebook account.

  - otherwise, associates this facebook account with that user and
    merges the new facebook info into that user's profile.

  - else, generates a new user and populates their profile using the
    supplied facebook info.

  (see signup/get-or-create-via-facebook! for impl details)"
  [m :- {:access-token s/Str
         :facebook-id s/Str
         (s/optional-key :id) as/ID}])

(s/defn unregister-facebook! :- as/User
  "Removes Facebook info from the supplied user. Returns that user."
  [id :- as/ID])

;; ## Get

(s/defn multiget :- {s/Any (s/either as/User as/Err)}
  "Returns a map of lookup value -> the user, if found. NOTE that this
  will fuck up if you send in the same value with different
  types. Also address error handling if a given value results in
  multiple users."
  [m :- [{:value s/Any, :type as/LookupType}]])

;; ## Update

(s/defn update! :- (s/either as/User as/Err)
  "Updates the user. If a new value for password, email or username is
  passed in, updates those fields appropriately (password gets hashed,
  email verification gets sent). Errors if the user doesn't exist."
  [id :- as/ID m :- {s/Keyword s/Any}])

;; ## Authentication

(s/defn trigger-password-reset! :- (s/either (s/eq nil) as/Err)
  "Generate a password reset and activation code for the supplied
  user. Errors if the user doesn't exist."
  [id :- as/ID])
;;TODO: Function that takes reset-code and new password, and tries to
;;reset. Also - email.

(s/defn reset-password! :- (s/either (s/eq nil) as/Err)
  ""
  [id :- as/ID
   reset-code :- s/Str
   new-password :- s/Str])

(s/defn password-valid? :- (s/either s/Bool as/Err)
  "Does the user with the supplied ID's password match this password?
  Errors if the password isn't set."
  [id :- as/ID password :- s/Str])

;; ## User Merge

(s/defn merge-users! :- (s/either as/User as/Err)
  "Merges the sub-id user into the primary-id user. Returns the merged
  user. Errors if either user doesn't exist.

  This can be used to merge real users, OR to merge pending users
  together."
  [primary-id :- as/ID
   sub-id :- as/ID])

;; ## Missing!
;;
;; - profile updates and gets
