(ns hub.user.rethink
  "Helpers for RethinkDB. Candidates for a shared library if other
  services start to use Rethink."
  (:require [rethinkdb.core :as rc]
            [rethinkdb.query :as r]
            [schema.core :as s]))

(s/defschema RethinkSpec
  {(s/optional-key :host) s/Str
   (s/optional-key :port) s/Int
   (s/optional-key :token) s/Str
   (s/optional-key :auth-key) s/Str
   (s/optional-key :db-name) (s/named s/Str "For User service only.")})

(defonce conf
  (atom {}))

(defn db-name
  "Returns the db-name for the current connection. (This assumes only
  one database is in use.)"
  []
  (:db-name @conf))

(defn connect
  "Same as redthinkdb.core/connect, except with our configs passed
  in. Configs should be initialized via a lifecycled compoment."
  []
  (apply rc/connect (apply concat @conf)))

(defn run
  "Runs a command wrapped in a connection using conf."
  [command]
  (with-open [c (connect)]
    (r/run command c)))

(s/defn try-create
  "Attempts to create a database with the supplied name. If it already
  exists, returns true; otherwise calls db-create."
  [db-name :- s/Str]
  (r/branch (r/contains (r/db-list) db-name)
            true
            (r/db-create db-name)))

(s/defn try-table
  "Attempts to create a table with the supplied name in the supplied
  database. If it already exists, returns true; otherwise calls
  table-create."
  [db-name :- s/Str
   table-name :- s/Str]
  (let [db (r/db db-name)]
    (r/branch (-> (r/table-list db)
                  (r/contains table-name))
              true
              (r/table-create db table-name))))

(defn get-by-id
  "Gets the document at the given id (primary key) in the table."
  [table id]
  (run (r/get table id)))
