(ns hub.user.client
  "User Service client."
  (:require [clj-time.core :as ct]
            [clojure.string :as str]
            [com.stuartsierra.component :as c]
            [hub.user.api.schema :as as]
            [hub.user.facebook :as fb]
            [hub.user.oauth :as oauth]
            [hub.user.schema :as us]
            [hub.user.service :as u]
            [hub.user.rethink :refer [RethinkSpec]]
            [hub.user.transforms :as t]
            [hub.queue.client :as qc]
            [hub.user.util :refer [encrypt compare-hash unix-time]]
            [schema.core :as s])
  (:import [com.stuartsierra.component Lifecycle]
           [java.util UUID]))

;; ## Create

;;TODO: Put create events on the queue.
(s/defn create! :- (s/either as/FullUser as/Err)
  "Generates a user off of the supplied email and password."
  [m :- as/SignupFields]
  (if (s/check as/SignupFields m)
    {:error {:type "invalid-input"}}
    (t/fulluser-model->fulluser-api (u/create! :signup m))))

(s/defn create-pending! :- (s/either as/PendingUser as/Err)
  "Generates a pending user instance. Returns an error if a user
  exists with the supplied email."
  [m :- as/PendingUserInput]
  (if (s/check as/PendingUserInput m)
    {:error {:type "invalid-input"}}
    (t/pendinguser-model->pendinguser-api (u/create! :pending m))))

;; ## Facebook
;;
;; These are pretty easy to extend to the other oauth providers.

(declare get-user)

(s/defn register-facebook! :- (s/either as/FullUser as/Err)
  "Returns a user off of the supplied facebook information. Optionally
   takes a UserID to associate the FaceBook info with.

  - If a user already exists with that facebook id, upgrades their
    token.

  - errors if another user is using the facebook account's email and
    ALREADY has another linked facebook account.

  - otherwise, associates this facebook account with that user and
    merges the new facebook info into that user's profile.

  - else, generates a new user and populates their profile using the
    supplied facebook info.

  (see signup/get-or-create-via-facebook! for impl details)"
  [m :- {:access-token s/Str
         :facebook-id s/Str
         (s/optional-key :id) as/ID}]
  (if-let [id (:id m)]
    (if-let [user (u/get-user-by-id id)]
      (-> (fb/attach-facebook! user m)
          (fb/attempt "attach-error"))
      {:error {:type "user-not-found"}})
    (-> (fb/get-or-create-via-facebook! m)
        (fb/attempt "facebook-error"))))

(s/defn unregister-facebook! :- as/User
  "Removes Facebook info from the supplied user. Returns that user."
  [id :- as/ID]
  (if-let [user (u/get-user-by-id id)]
    (t/fulluser-model->fulluser-api
     (oauth/unregister! user :facebook))
    {:error {:type "user-not-found"}}))

;; ## Get

(def LookupValue
  s/Any)

(def lookup-fns
  {:id u/get-users-by-ids
   :email u/get-users-by-emails
   :username u/get-users-by-usernames
   :name u/get-users-by-names})

(s/defn ^:private get-for-type :- {LookupValue [as/User]}
  "Returns all matches after doing a search of LookupType for the given vals."
  [type :- as/LookupType
   vals :- [LookupValue]]
  (let [{:keys [full pending]} ((lookup-fns type) vals)
        ;;:full and :pending each always have
        ;;{lookupvalue user} OR {lookupvalue [user]}
        convert-users (fn [type users]
                        (map (if (= type :full)
                               t/fulluser-model->fulluser-api
                               t/pendinguser-model->pendinguser-api)
                             users))
        transform (fn [[lookup-v user] type]
                    [lookup-v (convert-users type (if (or (list? user) (vector? user))
                                                    user
                                                    [user]))])
        ;;fulls and pendings are [lookup-v [converted-users]]
        fulls  (map transform full (repeat :full))
        pendings  (map transform pending (repeat :pending))]
    (into {} (concat fulls pendings))))

(s/defn multiget :- {LookupValue [as/User]}
  "Returns the collection of users that mach the given query. Email,
  username, and name lookups are downcased - so returned LookupValues
  are downcased for these types as well. Should NOT use the same
  LookupValue across multiple types, or this will
  non-deterministically pick one."

  ([ms :- [{:type as/LookupType
            :value LookupValue}]]
   (let [type->vals (reduce (fn [res {:keys [type value]}]
                              (assoc res type (conj (get res type []) value)))
                            {} ms)
         ;;results is a sequence of lookup-v->users
         results   (map (fn [[type vals]]
                          (get-for-type type vals))
                        type->vals)]
     (apply merge results)))

  ([type :- as/LookupType
    vals :- [LookupValue]]
   (multiget (for [v vals] {:type type :value v}))))

(s/defn get-user :- (s/maybe as/User)
  ([id :- as/ID] (get-user :id id))
  ([type :- as/LookupType v :- LookupValue]
   (-> (multiget [{:type type :value v}])
       (get v)
       (first))))

;; ## Update
(s/defn ^:private build-profile :- as/Profile
  "Creates a profile update"
  [{:keys [first-name last-name] :as m} :- as/UserUpdate]
  (merge (select-keys m [:gender :phone :location :birthdate :photos])
         (when-let [name-map (merge (when first-name
                                      {:first first-name})
                                    (when last-name
                                      {:last last-name}))]
           {:name name-map})))

(s/defn ^:private build-update :- as/User
  "Builds an update for the given user, from the UserUpdate."
  [user-to-update :- us/User
   user-type :- (s/enum :full :pending)
   {:keys [name username password email-address] :as m} :- as/UserUpdate]
  (let [full? (= user-type :full)
        profile-updates (build-profile m)]
    (merge (when (not-empty profile-updates)
             {:profile profile-updates})
           ;; Pending users only:
           (when (and (not full?) (not-empty name))
             {:name (:name m)})
           ;; Full users only:
           (when (and full? (not-empty username))
             {:username (:username m)})
           (when (and full? (not-empty password))
             {:password (encrypt password)})
           ;; Email:
           (when (not-empty email-address)
             {:email {:address email-address
                      :verified? (if (= (str/lower-case email-address)
                                        (-> user-to-update
                                            :email
                                            :address
                                            str/lower-case))
                                   (:verified? (:email user-to-update))
                                   false)}}))))

(s/defn update! :- (s/either as/User as/Err)
  "Updates the user. If a new value for password, email-address or a
  first or last name is passed in, updates those fields
  appropriately (password gets hashed). Errors if the user doesn't
  exist, otherwise returns the user with updates."
  ;;TODO: Transmit changes on queue (only if email got changed? - so
  ;;mailer can send verification.)
  [id :- as/ID
   {:keys [username email-address] :as m} :- as/UserUpdate]
  (if (s/check as/UserUpdate m)
    {:error {:type "invalid-input"}}
    (let [{:keys [full pending]} (u/get-users-by-ids [id])]
      (if (or (not-empty full) (not-empty pending))
        (let [user-type (if (not-empty full)
                          :full
                          :pending)
              user-to-update (or (get full id) (get pending id))
              lowercase-equals #(= (str/lower-case %1)
                                   (str/lower-case %2))
              check-needed? (fn [user s1 s2]
                              (and (= user-type :full)
                                   (when (and (not-empty s1) (not-empty s2))
                                     (not (lowercase-equals s1 s2)))))
              username-taken? (fn [user] (when (check-needed? user
                                                             username
                                                             (:username user-to-update))
                                          (-> (u/get-users-by-usernames [username])
                                              :full
                                              (get (str/lower-case username)))))
              email-taken? (fn [user] (when (check-needed? user
                                                          email-address
                                                          (:address
                                                           (:email user-to-update)))
                                       (-> (u/get-users-by-emails [email-address])
                                           :full
                                           (get (str/lower-case email-address)))))]
          (cond
            (username-taken? user-to-update) {:error {:type "username-taken"}}
            (email-taken? user-to-update) {:error {:type "email-taken"}}
            :else (if-let [created (->> (build-update user-to-update user-type m)
                                        (u/update-by-id! id))]
                    (if (= user-type :full)
                      (t/fulluser-model->fulluser-api created)
                      (t/pendinguser-model->pendinguser-api created))
                    {:error {:type "update-error"}})))
        {:error {:type "user-not-found"}}))))

(s/defn put-profile! :- (s/either as/User as/Err)
  "Put the given user-doc (replace the old one). Cant change id. Use
  this if you want to dissoc a field."
  [id :- s/Str
   profile :- as/Profile]
  (let [{:keys [full pending]} (u/get-users-by-ids [id])
        full-user (get full id)
        pending-user (get pending id)
        invalid-profile? #(s/check as/Profile %)
        put-user! (fn [user transform]
                    (-> (assoc user :profile profile)
                        u/put!
                        transform))]
    (cond
      (invalid-profile? profile) {:error {:type "invalid-input"}}
      full-user (put-user! full-user t/fulluser-model->fulluser-api)
      pending-user (put-user! pending-user t/pendinguser-model->pendinguser-api)
      :else {:error {:type "user-not-found"}})))

(s/defn delete! :- (s/either {:deleted (s/eq true)}
                             as/Err)
  "Returns true if delete succeeded, false otherwise."
  [id :- as/ID]
  (if (u/delete-by-id! id)
    {:deleted true}
    {:error {:type "delete-failed"}}))

;; ## Authentication

(s/defn trigger-password-reset! :- (s/either (s/eq nil)
                                             as/Err)
  "Generate a password reset and activation code for the supplied
  user. Errors if the user doesn't exist."
  [id :- as/ID]
  (let [reset-code (UUID/randomUUID)
        created-at (unix-time)]
    (when (nil? (u/update-by-id! id {:password-reset {:code reset-code
                                                      :created-at created-at}}))
      {:error {:type "trigger-password-reset-failed"}})
    ;;TODO: put this on queue.
    ))

(s/defn reset-password! :- (s/either (s/eq nil) as/Err)
  "If the reset-code is valid for the giver user-id, will set save a
  hashed version of their new-password."
  [reset-code :- s/Str
   new-password :- s/Str]
  ;;TODO: Put this on queue.
  (let [{:keys [password-reset] :as user} (u/get-user-by-reset-code reset-code)
        two-days (ct/in-millis (ct/days 2))]
    (if (= (:code password-reset) reset-code)
      (if (> two-days (- (unix-time) (:created-at password-reset)))
        (when (nil? (u/put! (-> user
                                (dissoc :password-reset)
                                (assoc :password (encrypt new-password)))))
          {:error {:type "error-updating-password"}})
        {:error {:type "expired-reset-code"}})
      {:error {:type "incorrect-reset-code"}})))

(s/defn password-valid? :- (s/either s/Bool as/Err)
  "Does the user with the supplied ID's password match this password?
  Errors if the password isn't set."
  [id :- as/ID
   password :- s/Str]
  (if-let [user (get (:full (u/get-users-by-ids [id])) id)]
    (if (not-empty (:password user))
      (compare-hash password (:password user))
      {:error {:type "password-not-set"}})
    {:error {:type "user-not-found"}}))

;; ## User Merge

(s/defn merge-users! :- (s/either as/User as/Err)
  "Merges the sub-id user into the primary-id user. Returns the merged
  user. Errors if either user doesn't exist.

  This can be used to merge full users, OR to merge pending users
  together or into full users. Can't merge full users into a pending
  user."
  [primary-id :- as/ID
   sub-id :- as/ID]
  ;;TODO: Put something on the queue when this happens. Think about
  ;;coordination with other services. Ideally we would know if an
  ;;error occured in any service's merge-user implementation.
  (if-let [merged (u/merge-users! primary-id sub-id)]
    (if (= (u/user-type merged) :full)
      (t/fulluser-model->fulluser-api merged)
      (t/pendinguser-model->pendinguser-api merged))
    {:error {:type "merge-user-error"}}))

;; ## API Initialization

(s/defn user-client :- Lifecycle
  "Client for the user service."
  [opts :- {:rethink {:mode (s/enum :test :live)
                      :db-name s/Str
                      :spec RethinkSpec}}]
  (let [{:keys [mode db-name spec]} (:rethink opts)]
    (u/rethinkdb mode db-name spec)))
