(ns hub.user.facebook
  "Facebook API helpers."
  (:require [clojure.core.async :as a]
            [clojure.string :as str]
            [cheshire.core :refer [parse-string]]
            [environ.core :as e]
            [org.httpkit.client :as http]
            [hub.user.api.schema :as as]
            [hub.user.config :as c]
            [hub.user.oauth :as oauth]
            [hub.user.schema :as us]
            [hub.user.service :as service]
            [hub.user.transforms :as t]
            [hub.util.aws :as aws]
            [ring.util.codec :as ring-codec]
            [schema.core :as s])
  (:import [java.io ByteArrayInputStream]))

;; This namespace requires FB_APP_SECRET, FB_CLIENT_ID, and
;; FB_CLIENT_SECRET environment variables.

;; ## Schema

(s/defschema CallBack
  {:path s/Str
   :domain s/Str})

(s/defschema TokenLocation
  (s/enum :params :body))

(s/defschema OAuthConfig
  {:token-location TokenLocation
   :auth-url s/Str
   :token-url s/Str
   :client-id s/Str
   :client-secret s/Str
   (s/optional-key :auth-query) {s/Keyword s/Str}
   (s/optional-key :callback) CallBack})

(s/defschema APIError
  "Error response from the Facebook API."
  {:error {:message s/Str :type s/Str :code s/Int}})

(s/defschema AuthMap
  {:facebook-id s/Str
   :access-token s/Str})

(s/defschema FacebookConfig
  {:oauth OAuthConfig
   :app-secret s/Str})

;; ## Facebook API Data

(def api-version "2.2")
(def root (str "https://graph.facebook.com/v" api-version "/"))

(s/defn facebook-config :- FacebookConfig
  "Token location specifies that the token is going to come back in
  the params, not the body. We also make sure to ask for email
  privileges, to beef up a particular user's profile.
  The returned map contains oauth information AND the app-specific
  secret key for facebook."
  []
  {:app-secret (c/env :fb-app-secret)
   :oauth {:token-location :params
           :auth-url "https://www.facebook.com/dialog/oauth"
           :token-url "https://graph.facebook.com/oauth/access_token"
           :auth-query {:scope "email", :response_type "code"}
           :client-id (c/env :fb-client-id)
           :client-secret (c/env :fb-client-secret)}})

(def app-id (comp :client-id :oauth facebook-config))
(def client-secret (comp :client-secret :oauth facebook-config))
(def server-secret (comp :app-secret facebook-config))

(s/defn body
  "returns a json-parsed representation of the body of the response."
  [req]
  (parse-string (:body @req) keyword))

(defn get-access-token-from-params
  "Alternate function to allow retrieve
  access_token when passed in as form params."
  [{body :body}]
  (-> body ring-codec/form-decode (get "access_token")))

(s/defn access-token :- (s/maybe s/Str)
  "Generates a server-side access token. This is used to make calls
  about the app to the server.
  More info:
  https://developers.facebook.com/docs/facebook-login/access-tokens"
  []
  (let [params {:client_id (app-id)
                :client_secret (client-secret)
                :grant_type "client_credentials"}]
    (get-access-token-from-params
     @(http/get (str root "oauth/access_token")
                {:query-params params}))))

(s/defschema GetOpts
  {(s/optional-key :query-params) {s/Any s/Any}
   (s/optional-key :params) {s/Any s/Any}})

(s/defn api-get :- {s/Any s/Any}
  "Makes a call to Facebook's /user/id endpoint for the user linked to
  the supplied token. Gives you a map with a bunch of information
  about the user.
  See the documentation for more details on what the map holds:
  https://developers.facebook.com/docs/graph-api/reference/v2.1/user"
  [endpoint :- s/Str
   {:keys [query-params params]} :- GetOpts]
  (let [opts (assoc params :query-params query-params)]
    (body (http/get (str root endpoint) opts))))

;; ## Api Calls

;; ### Token Exchange
;;
;; This section deals with exchanging short-lived client tokens for
;;long-lived server tokens.

(s/defn exchange-token :- (s/either APIError {:token s/Str})
  "Accepts a client-side token from the Facebook JS SDK and exchanges
  it for a long-lived token. The long token lasts for about 60 days
  from last use.
  If the user logs in with facebook, or uses our site and our JS SDK
  refreshes them, the token renews for another sixty days.
  The client side token by itself is only good for an hour or so."
  [token :- s/Str]
  (let [{:keys [client-id client-secret]} (:oauth (facebook-config))
        resp @(http/get (str root "oauth/access_token")
                        {:query-params
                         {:grant_type "fb_exchange_token"
                          :fb_exchange_token token
                          :client_id client-id
                          :client_secret client-secret}})]
    (or (when-let [token (get-access-token-from-params resp)]
          {:token token})
        (parse-string (:body resp) keyword))))

;; ### Profile Info

(s/defschema UserData
  "Response from Facebook's me endpoint. The keys here are really just
  examples of the fields that can come through. If you ask for
  specific fields, the rest will be missing, and certain users will
  just not have certain fields."
  {(s/optional-key :email) s/Str
   (s/optional-key :first_name) s/Str
   (s/optional-key :timezone) (s/named s/Int "UTC offset.")
   (s/optional-key :locale) s/Str
   (s/optional-key :name) s/Str
   (s/optional-key :updated_time) s/Str
   (s/optional-key :link) s/Str
   (s/optional-key :id) s/Str
   (s/optional-key :last_name) s/Str
   (s/optional-key :gender) s/Str
   (s/optional-key :verified) s/Bool
   s/Any s/Any})

(s/defn me :- UserData
  "Makes a call to Facebook's /user/id endpoint for the user linked to
  the supplied token. Gives you a map with a bunch of information
  about the user.
  Optionally you can supply a list of fields that you want
  returned. If you don't supply any, you get a grab-bag of fields.
  See the documentation for more details on what the map holds:
  https://developers.facebook.com/docs/graph-api/reference/v2.1/user"
  ([token :- s/Str]
   (me token nil))
  ([token :- s/Str fields :- [s/Str]]
   (api-get "me" {:query-params {:access_token token
                                 :fields (str/join "," fields)}})))

(s/defschema CoverPhoto
  {:id (s/named s/Str "ID of the photo in Facebook.")
   (s/optional-key :offset_y) s/Int
   (s/optional-key :source) (s/named s/Str "Photo URL.")})

(s/defschema ProfilePhoto
  {:is_silhouette s/Bool
   :url (s/named s/Str "URL of the photo on Facebook's CDN.")
   (s/optional-key :width) s/Int
   (s/optional-key :height) s/Int})

(s/defn cover-photo :- CoverPhoto
  "Returns information for the supplied user's cover photo. If the
  user's cover photo isn't set, only the :id is present."
  [token :- s/Str]
  (:cover (me token ["cover"])))

(s/defn largest-cover
  "Takes in a FB cover photo and queries the API for the highest
  resolution version of that photo."
  [cover :- CoverPhoto]
  (let [{:keys [height width source]}
        (->> (api-get (:id cover)
                      {:query-params {:type "large" :redirect false}
                       :params {:oauth-token (server-secret)}})
             :images
             (sort-by (complement :width))
             (first))]
    {:url source
     :y-offset (:offset_y cover)
     :height height
     :width width}))

(s/defschema ProfilePhotoOpts
  "Options for the profile-photo API call."
  {:user-id s/Str
   (s/optional-key :width) s/Int
   (s/optional-key :height) s/Int})

(s/defn profile-photo :- (s/maybe ProfilePhoto)
  "If the supplied user exists, returns information for the supplied
  user's cover photo. If the user's cover photo isn't set, only
  the :id is present."
  [opts :- ProfilePhotoOpts]
  (:data
   (api-get (str (:user-id opts) "/picture")
            {:query-params (merge (select-keys opts [:width :height])
                                  {:type "large" :redirect false})
             :params {:oauth-token (server-secret)}})))

;; ## Photo Processing

(s/defn url->photo :- aws/Image
  "For users, the `photo-name` passed to AWS has been main-photo or
  logo."
  [url :- s/Str]
  (let [{:keys [status headers body error]} @(http/get url {:as :stream})]
    (-> (select-keys headers [:content-length :content-type])
        (assoc :tempfile body))))

;; For users, the `photo-name` passed to AWS has been main-photo or
;;  logo.
(s/defn facebook->aws :- {:success s/Bool
                          :url s/Str}
  "Attempts to transfer the supplied facebook image to AWS. Returns
    a map with :success true if the transfer was successful, false
    otherwise. Either way the result will have the proper URL to
    store.

   If the skip-aws env variable is set, skips upload."
  [facebook-url :- s/Str
   image-type :- (s/enum "main-photo" "logo")]
  (if (not-empty (e/env :skip-aws))
    {:success true :url facebook-url}
    (let [conf (aws/aws-config)
          cloudfront (-> conf :buckets :photos :cloudfront)
          photo (url->photo facebook-url)
          chan (aws/upload-image conf :photos image-type photo)
          {:keys [success url]} (a/<!! chan)]
      {:success success
       :url (if success
              (str cloudfront "/" url)
              facebook-url)})))

;; # Service Methods

(s/defn parse-birthdate :- us/BirthDate
  "Grabs the birthdate object from Facebook's birthday field."
  [fb-birthdate :- s/Str]
  (let [parse-long (fn [x] (if (number? x) x (Long/parseLong x)))
        [month day year] (str/split fb-birthdate #"/")]
    {:month (parse-long month)
     :day (parse-long day)
     :year (parse-long year)}))

(s/defn parse-photos :- (s/maybe {us/ImageType us/ImageDetails})
  "Parses the photos returned by Facebook's API."
  [profile :- ProfilePhoto
   cover :- CoverPhoto]
  (letfn [(aws-xfer [image-type]
            (fn [url] (:url (facebook->aws url image-type))))]
    (not-empty
     (merge (when-not (:is_silhouette profile)
              {:profile (-> (select-keys profile [:url :width :height])
                            (update :url (aws-xfer "logo")))})
            (when-let [source (:source cover)]
              {:cover (-> (largest-cover cover)
                          (update :url (aws-xfer "main-photo")))})))))

(s/defn generate-profile :- us/Profile
  [profile-photo :- ProfilePhoto
   me-response :- UserData]
  (let [{:keys [first_name last_name birthday cover gender]} me-response]
    (merge
     {:name {:first first_name, :last last_name}}
     (when-let [bd (and birthday (parse-birthdate birthday))]
       {:birthdate bd})
     (when-let [photos (parse-photos profile-photo cover)]
       {:photos photos})
     (when-let [gender (#{"male" "female"} gender)]
       {:gender gender}))))

(def racehub-fb-fields
    "Fields we need to ask for when we query Facebook."
    ["birthday" "cover"
     "first_name" "last_name" "name"
     "email" "gender" "verified"])

(s/defschema FacebookMeta
  "Facebook Metadata we store in RH."
  {:name s/Str
   :email s/Str
   :id s/Str})

(s/defmethod oauth/oauth-metadata :facebook :- FacebookMeta
  [auth-type :- oauth/OAuthProvider token :- s/Str]
  (-> (me token)
      (select-keys [:proname :email :id])))

(s/defmethod oauth/oauth-name :facebook :- s/Str
  [auth-type :- oauth/OAuthProvider
   metadata :- FacebookMeta]
  (:name metadata))

(s/defmethod service/generate-user :facebook
  [_ {token :access-token id :facebook-id :as m}]
  (let [token (future (:token (exchange-token token)))
        prof-photo (future (profile-photo {:user-id id, :width 400, :height 400}))
        me (future (me @token ["birthday" "cover"
                               "first_name" "last_name" "name"
                               "email" "gender" "verified"]))
        {:keys [email first_name last_name verified]} @me]
    {:username (service/generate-username
                {:email email
                 :first-name first_name
                 :last-name last_name})
     :email {:address email
             :verified? verified}
     :profile (generate-profile @prof-photo @me)
     :oauth {:facebook {:token @token
                        :metadata (select-keys @me [:name :email :id])}}}))

(s/defn upgrade-token! :- us/User
    "Takes an existing user and a client side token and attempts to
    upgrade that token to a new, server side token using the Facebook
    API."
    [user :- us/User access-token :- s/Str]
    (if-let [upgraded (:token (exchange-token access-token))]
      (service/update! user {:oauth {:facebook {:token upgraded}}})
      user))

(s/defn merge-facebook :- us/User
  "Merges the generated Facebook user into the current user doc. The
    rule is that keys that already exist in the user's profile (Except
    for the new facebook token) win."
  [user :- us/User facebook-user :- us/FullUserInput]
  (letfn [(merge* [l r]
            (if (map? l)
              (merge-with merge* l r)
              l))]
    (merge-with merge* user facebook-user)))

(s/defschema FBResponse
  (s/either
   {:ok (s/pred true?)
    :user s/Any
    :created? s/Bool}
   {:ok (s/pred false?)
    (s/optional-key :reason) s/Str}))

(s/defn attach-facebook! :- FBResponse
  [user :- us/FullUser m :- AuthMap]
  (if-let [token (:token (exchange-token (:access-token m)))]
    (let [updated (-> (merge-facebook user (service/generate-user :facebook m))
                      (oauth/register-auth! :facebook token))]
      {:ok true :user updated :created false})
    {:ok false :reason "Token exchange failed."}))

(s/defn get-or-create-via-facebook! :- FBResponse
  "If the user exists, updates its facebook token and returns the
    updated user. If the user does NOT exist, creates the user, sends
    a welcome email and returns that."
  [{:keys [facebook-id access-token] :as m} :- AuthMap]
  (or (when-let [existing (service/get-user-by-fb-id facebook-id)]
        {:ok true
         :created? false
         :user (upgrade-token! existing access-token)})
      (let [fb-input (service/generate-user :facebook m)]
        (if-let [other (:fulluser
                        (service/get-user-by-email
                         (-> fb-input :email :address)))]
          (if (oauth/oauth-registered? other :facebook)
            {:ok false, :reason "Other user already has a linked FB."}
            {:ok true
             :created? false
             :user (service/put! (merge-facebook other fb-input))})
          (let [new-user (service/create! fb-input)]
            {:ok true, :created? true, :user new-user})))))

(s/defn attempt :- (s/either as/FullUser as/Err)
  "Helps with generating API responses off of FBResponse
  objects. Returns an API user."
  [resp :- FBResponse error-reason :- s/Str]
  (if (:ok resp)
    (t/fulluser-model->fulluser-api (:user resp))
    {:error {:type error-reason, :message (:reason resp)}}))
