(ns hub.user.service
  "Service implementation details."
  (:require [clojure.string :as str]
            [com.stuartsierra.component :as c]
            [hub.user.config :refer [fulluser-table-name pendinguser-table-name]]
            [hub.user.schema :as us]
            [hub.user.setup :refer [setup!]]
            [hub.user.util :as su]
            [hub.util.rethink :as ur]
            [rethinkdb.query :as r]
            [schema.core :as s])
  (:import [com.stuartsierra.component Lifecycle]))

;; ## Helpers

(defn fulluser-table
  "Returns a command for the full user table, suitable for chaining
  with other commands."
  []
  (r/table (r/db (ur/db-name)) fulluser-table-name))

(defn pendinguser-table
  "Returns a command for the pending user table, suitable for chaining
  with other commands."
  []
  (r/table (r/db (ur/db-name)) pendinguser-table-name))

(defmacro run-fulluser
  "Runs a command on the user table."
  [& commands]
  `(ur/run (-> (fulluser-table)
               ~@commands)))

(defmacro run-pendinguser
  "Runs a command on the user table."
  [& commands]
  `(ur/run (-> (pendinguser-table)
               ~@commands)))

;; ## GET API

(s/defn get-all-full-users :- [us/FullUser]
  []
  (run-fulluser))

(s/defn get-all-pending-users :- [us/PendingUser]
  []
  (run-pendinguser))

;; ## Single Gets:
(s/defn get-user-by-id :- (s/maybe us/User)
  "Returns the user for the given id. Could be a full user or pending user."
  [id :- us/UserID]
  (when-let [match (ur/run
                     (r/do [(r/get (fulluser-table) id)
                            (r/get (pendinguser-table) id)]
                           (r/fn [arg]
                             (r/or (r/nth arg 0) (r/nth arg 1)))))]
    match))

(s/defn get-user-by-email :- {:fulluser (s/maybe us/FullUser)
                              :pending [us/PendingUser]}
  "Returns all matching full users and pending users for the given email."
  [email-address :- s/Str]
  {:fulluser (-> (r/get-all [email-address] {:index "email"})
                 (run-fulluser)
                 first)
   :pending (run-pendinguser
             (r/get-all [email-address] {:index "email"}))})

(s/defn get-user-by-reset-code :- (s/maybe us/FullUser)
  [reset-code :- s/Str]
  (first (run-fulluser (r/get-all [reset-code] {:index "pw-reset-code"}))))

(s/defn get-user-by-fb-id :- (s/maybe us/FullUser)
  "Does a lookup for a FullUser by facebook id."
  [fb-id :- s/Str]
  (first (run-fulluser (r/get-all [fb-id] {:index "facebook-id"}))))

;; ## Bulk Gets:
(def LookupValue s/Str)

(defn map-by [key-func val-func coll]
  (into {} (map (juxt key-func val-func) coll)))

(defn map-values
  "Maps the keyspace using the supplied function. Any duplicate keys
  will get knocked out in a nondeterministic order, so be careful!"
  [f m]
  (into {} (for [[k v] m]
             [k (f v)])))

(s/defn get-users-by-ids :- {:full {us/UserID us/FullUser}
                             :pending {us/UserID us/PendingUser}}
  "Returns all users that match the given ids. Checks FullUsers and
  PendingUsers. Only one match per ID."
  [ids :- [us/UserID]]
  (let [result (ur/run (r/do [(r/coerce-to (r/get-all (fulluser-table) ids) "ARRAY")
                              (r/coerce-to (r/get-all (pendinguser-table) ids) "ARRAY")]
                             (r/fn [args]
                               (r/object "full" (r/nth args 0)
                                         "pending" (r/nth args 1)))))]
    (map-values #(map-by :id identity %) result)))

(s/defn try-lower-case :- (s/maybe s/Str)
  [s :- (s/maybe s/Str)]
  (when s
    (str/lower-case s)))

(s/defn get-users-by-emails :- {:full {us/EMailAddress us/FullUser}
                                :pending {us/EMailAddress [us/PendingUser]}}
  "Returns all users that match the given emails, lower cased. Checks
  FullUsers (one match per email) and PendingUsers (might be multiple
  matches). Returns lowercased emails to Users."
  [emails :- [us/EMailAddress]]
  (let [emails (map try-lower-case emails)
        result (ur/run (r/do [(r/coerce-to (r/get-all (fulluser-table) emails {:index "email"})
                                           "ARRAY")
                              (r/coerce-to (r/get-all (pendinguser-table) emails
                                                      {:index "email"})
                                           "ARRAY")]
                             (r/fn [args]
                               (r/object "full" (r/nth args 0)
                                         "pending" (r/nth args 1)))))]
    {:full (map-by (comp try-lower-case :address :email) identity (:full result))
     :pending (group-by (comp try-lower-case :address :email) (:pending result))}))

(s/defn get-users-by-usernames :- {:full {us/UserName us/FullUser}}
  "Returns the usernames for the given users. No pending users, each
  username only matches to one fulluser. Does a lowercase=
  match. Returns lowercased usernames to FullUser."
  [usernames :- [us/UserName]]
  (let [usernames (map try-lower-case usernames)
        full (run-fulluser
              (r/get-all usernames {:index "username"})
              (r/coerce-to "ARRAY"))]
    {:full (map-by (comp try-lower-case :username) identity full)}))

(s/defn get-users-by-names :- {:full {us/FullName [us/FullUser]}
                               :pending {us/FullName [us/PendingUser]}}
  "Returns those users (full and pending) that match the given name
  query with whitespace removed, and lowercased. We return downcased
  FullName to matches."
  [names :- [us/FullName]]
  (let [names (set (map (comp #(str/replace % " " "") str/lower-case) names))
        result (ur/run (r/do [(r/coerce-to (r/get-all (fulluser-table) names {:index "name"})
                                           "ARRAY")
                              (r/coerce-to (r/get-all (pendinguser-table) names
                                                      {:index "name"})
                                           "ARRAY")]
                             (r/fn [args]
                               (r/object "full" (r/nth args 0)
                                         "pending" (r/nth args 1)))))]
    ;;TODO: Should this return `names` the same way they came in? Or
    ;;like this? Or all lowercased and spaces removed?
    {:full (group-by (fn [{:keys [profile]}]
                       (try-lower-case (str (:first (:name profile)) " "
                                            (:last (:name profile)))))
                     (:full result))
     :pending (group-by (comp try-lower-case :name) (:pending result))}))

;; ## User Generation
(s/defschema GenOptions
  "Options for username generation."
  {:email s/Str
   (s/optional-key :first-name) s/Str
   (s/optional-key :last-name) s/Str})

(s/defn winner :- (s/maybe s/Str)
  "Queries the database for all supplied username
    candidates. Returns the first candidate that doesn't already exist
    in the database, or nil if they all do."
  [candidates :- [(s/maybe s/Str)]]
  (let [candidates (->> candidates
                        (remove (some-fn empty? (complement su/username-valid?)))
                        (map str/lower-case))
        fetched (:full (get-users-by-usernames candidates))]
    (some (fn [k] (when-not (fetched k) k)) candidates)))

(s/defn generate-username :- s/Str
  "Generates a random username for the supplied GenOptions map of
    seeding options. Tries concatenating the first and last names,
    taking just the first initial and last name, and the email
    prefix (before the @ sign). After that, attempts to generate a
    username.  NOT guaranteed to terminate, but extremely likely :)"
  [{:keys [email first-name last-name]} :- GenOptions]
  (let [email-prefix            (first (str/split email #"@"))
        first-initial-last-name (str (first first-name) last-name)
        first-tries [email-prefix
                     first-initial-last-name
                     (str first-name last-name)]
        un-prefix (or email-prefix last-name)]
    (loop [candidates first-tries seed 1000]
      (or (winner candidates)
          (recur (take 10 (su/random-usernames un-prefix seed))
                 (* seed 10))))))

(defmulti generate-user
  "Generates a user document off of the information provided for the
  supplied type. Signup gives username and password only, for
  example. Facebook gives a bit more."
  (fn [type data] type))

(s/defmethod generate-user :signup :- (us/toggle-optional us/FullUserInput :password)
  [_ {:keys [email password]} :- us/SignupFields]
  {:password (su/encrypt password)
   :username (generate-username {:email email})
   :email {:address email, :verified? false}
   :profile {}})

(s/defmethod generate-user :pending :- us/PendingUserInput
  [_ {:keys [name email profile] :as pending} :- us/PendingUserInput]
  (let [profile (su/build-profile (or profile {}))]
    (merge (select-keys pending [:name :email])
           (when (not-empty profile)
             {:profile profile}))))

(s/defmethod generate-user :default :- us/FullUserInput
  [_ m :- us/FullUserInput]
  (cond-> m
    (:username m) (update :username str/lower-case)
    (-> m :email :address) (update-in [:email :address] str/lower-case)
    (:password m) (update :password su/encrypt)
    (:profile m) (update :profile su/build-profile)
    (:oauth m) (update :oauth (fn [oauth]
                                (if (s/check us/OauthMap oauth)
                                  {}
                                  oauth)))))

;; ## MUTATING API
(s/defschema OpError
  {:ok (s/eq false)
   (s/optional-key :reason) s/Str})

(s/defschema OpSuccess
  {:ok (s/eq true)})

(s/defschema OpStatus
  (s/either OpError OpSuccess))

(s/defschema UserType
  (s/enum :full :pending))

(s/defn user-type :- UserType
  "Returns the user type, depending on whether or not a :username is present."
  [user :- us/User]
  (if (:username user)
    :full
    :pending))

(s/defn email-and-username-ok? :- OpStatus
  "If the email or username is different in the proposed updates, it
  will check to see if they are allowed (for FullUsers). Email here is
  an EmailMap."
  [prev-user :- (s/named s/Any "previous user fragment.")
   type :- UserType
   updates :- (s/named s/Any "user fragment")]
  (if (= type :full)
    (let [check-needed? (fn [prev new]
                          (and (not (su/lowercase-equals prev new))
                               (not-empty new)))
          username-taken? (fn [username] (when (check-needed? (:username prev-user)
                                                             username)
                                          (-> (get-users-by-usernames [username])
                                              :full
                                              (get (str/lower-case username)))))
          email-taken? (fn [ea] (when (check-needed? (:address (:email prev-user))
                                                    ea)
                                 (-> (get-users-by-emails [ea])
                                     :full
                                     (get (str/lower-case ea)))))]
      (cond
        (username-taken? (:username updates))
        {:ok false
         :reason "Username is already taken!"}
        (email-taken? (:address (:email updates)))
        {:ok false
         :reason "Email is already taken!"}
        :else {:ok true}))
    {:ok true}))

;;TODO: What happens if a :facebook signup generates a taken email or
;;username? Make sure we have test coverage.
(s/defn create! :- (s/either us/User OpError)
  "Creates a full user document using the supplied method; returns the
  created FullUser or PendingUser. For non-pending users, given email
  and username (if specified) must be unique."
  ([data :- us/CreateInput]
   (create! :default data))
  ([type :- us/SignupType
    data :- us/CreateInput]
   (let [user (generate-user type data)
         base (merge user {:created-at (su/unix-time)})
         user-type (if (= type :pending)
                     :pending
                     :full)
         change-status (email-and-username-ok? {} user-type user)]
     (if (:ok change-status)
       (let [create-result (if (= type :pending)
                             (run-pendinguser (r/insert base))
                             (run-fulluser (r/insert base)))]
         (assoc base :id (first (:generated_keys create-result))))
       change-status))))

(s/defn update! :- (s/either us/User OpError)
  "Updates the user with the given id by merging it with the new
  map. Can't update id. Works for FullUsers and PendingUsers. Returns
  the record for the updated user, or nil if the user-id was
  invalid. Note that this is a deep merge - nested fields get merged
  in, other keys in the nested field are unaffected."
  [user :- (s/either us/UserID us/User)
   updates :- {s/Any s/Any}]
  ;;r/update: http://rethinkdb.com/api/javascript/update/
  (let [[prev-user user-id] (if (coll? user)
                              [user (:id user)]
                              [(get-user-by-id user) user])]
    (if (not-empty prev-user)
      (let [op-status (email-and-username-ok? prev-user (user-type prev-user) updates)]
        (if (:ok op-status)
          (let [updates (assoc updates :updated-at (su/unix-time))]
            (ur/run (r/do [(r/update (r/get (fulluser-table) user-id) updates)
                           (r/update (r/get (pendinguser-table) user-id) updates)]
                          (r/fn [arg]
                            arg)))
            (get-user-by-id user-id))
          op-status))
      {:ok false
       :reason "User does not exist!"})))

(s/defn put! :- (s/either us/User OpError)
  "Puts the user-doc with the desired changes. Cant change id. Email
  and username must be unique for FullUsers. Works for FullUsers and
  PendingUsers. Returns the newly put user, if the id was valid."
  [{:keys [id] :as new-user} :- us/User]
  ;;r/replace: http://rethinkdb.com/api/javascript/replace/
  (let [prev-user (get-user-by-id id)]
    (if (not-empty prev-user)
      (let [op-status (email-and-username-ok? prev-user (user-type prev-user) new-user)]
        (if (:ok op-status)
          (let [updated (merge new-user {:updated-at (su/unix-time)})
                {:keys [full pending]} (get-users-by-ids [id])]
            (when (not-empty (concat full pending))
              (-> (if (not-empty full)
                    (fulluser-table)
                    (pendinguser-table))
                  (r/get id)
                  (r/replace updated)
                  (ur/run))
              updated))
          op-status))
      {:ok false
       :reason "User does not exist!"})))

(s/defn delete-by-id! :- s/Bool
  "Deletes the given FullUser or PendingUser, returns true if the id
  was deleted, or false if not."
  [user-id :- us/UserID]
  ;;r/delete: http://rethinkdb.com/api/javascript/delete/
  (ur/run
    (r/do [(r/get-field (r/delete (r/get (fulluser-table) user-id))
                        :deleted)
           (r/get-field (r/delete (r/get (pendinguser-table) user-id))
                        :deleted)]
          (r/fn [args]
            (r/or (r/eq (r/nth args 0) 1)
                  (r/eq (r/nth args 1) 1))))))

;;TODO: put-bulk, update-bulk, delete-bulk

(s/defn ^:private build-name :- {:first s/Str
                                 :last s/Str}
  "Used if merging a pending user into a full user, to determine the
  proper name to use."
  [primary-user :- us/FullUser
   absorbed-user :- us/PendingUser]
  (let [full-name? (fn [user]
                     (and (not-empty (get-in user [:profile :name :first]))
                          (not-empty (get-in user [:profile :name :last]))))]
    (if (full-name? primary-user)
      ;; if the primary user has a full :profile :name, use it
      (get-in primary-user [:profile :name])
      (if (full-name? absorbed-user)
        ;; else if the absorbed user has a full :profile :name, use it
        (get-in absorbed-user [:profile :name])
        ;; else use :name from pending to build a :profile :name
        (let [names (str/split (:name absorbed-user) #" ")]
          {:first-name (first names)
           :last-name (str/join " " (rest names))})))))

(s/defn merge-users! :- (s/maybe us/User)
  "Merges absorbed user into primary user. Returns the primary with
  the newly merged fields. You can merge pending into pending, full
  into full, pending into full, but NOT full into pending."
  [{:keys [profile] :as primary-user-id} :- us/UserID
   absorbed-user-id :- us/UserID]
  (let [primary-user (get-user-by-id primary-user-id)
        absorbed-user (get-user-by-id absorbed-user-id)]
    (when (and primary-user absorbed-user)
      (let [primary-user-type (user-type primary-user)
            absorbed-user-type (user-type absorbed-user)
            new-user
            (cond
              ;;full into pending not allowed:
              (and (= primary-user-type :pending)
                   (= absorbed-user-type :full))
              nil
              ;; merging full into full, or pending into pending:
              (= primary-user-type absorbed-user-type)
              (su/deep-merge absorbed-user primary-user)
              ;;merging pending into full:
              :else
              (su/deep-merge (dissoc absorbed-user :name)
                             (assoc-in primary-user [:profile :name]
                                       (build-name primary-user absorbed-user))))
            created (when new-user
                      (put! new-user))]
        (when created
          (delete-by-id! absorbed-user-id)
          created)))))

;; Note: r/update, r/delete, r/replace can all work on multiple docs,
;;but the logic has to be build into the REQL query. For multiple ids,
;;just use r/get-all instead of r/get
