(ns hub.user.schema
  (:require [schema.core :as s]))

;; # User
;;
;; The user model's responsible for
;;
;; - Authentication
;; - identity across tickets, etc
;; - holds info about the user's profile

(def UserID s/Str)
(def UserName s/Str)

(def EMailAddress
  s/Str)

(s/defschema EMailMap
  {:address EMailAddress
   :verified? s/Bool
   (s/optional-key :activation-code) s/Str})

(def genders #{"male" "female"})
(def Gender (apply s/enum genders))

(def PasswordHash (s/named s/Str "Hash of the user's password."))

(s/defschema BirthDate
  {:day s/Int :month s/Int :year s/Int})

(def FullName
  s/Str)

(s/defschema Address
  {(s/optional-key :street-number) s/Str
   (s/optional-key :street-address) s/Str
   (s/optional-key :city) s/Str
   (s/optional-key :zip) s/Str
   (s/optional-key :state) s/Str
   (s/optional-key :country) s/Str})

(def Timezone s/Str)

(s/defschema Location
  {(s/optional-key :formatted-address) s/Str
   (s/optional-key :address) Address
   (s/optional-key :timezone) Timezone})

(s/defschema OAuthProvider
  (s/enum :facebook :strava :twitter))

(def ImageType
  "Identifier for the image, like 'cover' or 'profile'"
  s/Keyword)

(s/defschema ImageDetails
  {:url s/Str
   (s/optional-key :y-offset) (s/named s/Num "For cover photos, a
percentage between 0 and 100 that tells us how to vertically position
the photo. If not provided, we assume 0.5. This figure is relevant for
an image with AR less than the containing div's AR. This means that
when we fit the image width to the container width, some of the
top/bottom will overflow outside the div. The y-offset tells us what
percent of the overflow should be located above vs below the pic. Ex:
Image of height 500px, cover div height 300px, y-offset=0.4, means
200px cant fit vertically, margin-top=-50, so 50px of the top, and
150px of the bottom is hidden.")
   (s/optional-key :x-offset) (s/named s/Num "Same concept as
     y-offset (percentage between 0 and 100), but for an image whose
     AR is greater than the containing div's AR, meaning when we fit
     the image's height, some of the left/right spills outside the
     div. The offset tells us where to center the image horizontally
     in the div.")
   :height (s/named s/Int "Height of the original image.")
   :width (s/named s/Int "Width of the original image.")})

(s/defschema Profile
  {(s/optional-key :name) {:first s/Str :last s/Str}
   (s/optional-key :gender) Gender
   (s/optional-key :phone) s/Str
   (s/optional-key :location) Location
   (s/optional-key :birthdate) BirthDate
   (s/optional-key :photos) {ImageType ImageDetails}})

(s/defschema PendingProfile
  (assoc Profile
         (s/optional-key :name) {(s/optional-key :first) s/Str
                                 (s/optional-key :last) s/Str}))

(def UnixTime
  java.lang.Long)

(s/defschema RethinkFields
  {:id UserID
   :created-at UnixTime
   (s/optional-key :updated-at) UnixTime})

(s/defschema SignupFields
  {:email s/Str
   :password s/Str})

(s/defschema OauthMap
  {OAuthProvider
   {:token s/Str
    :id s/Str
    :metadata s/Any}})

(s/defschema FullUserInput
  {:username UserName
   :email EMailMap
   :profile Profile
   (s/optional-key :roles) [(s/enum "user" "superadmin")]
   (s/optional-key :oauth) OauthMap
   (s/optional-key :password) PasswordHash
   (s/optional-key :password-reset) {:code s/Str, :created-at UnixTime}})

(s/defschema FullUser
  "Full User schema."
  (merge FullUserInput
         RethinkFields))

(s/defschema PendingUserInput
  "User record for a user without an authentication. Might be
  duplicates."  {:name (s/named s/Str "Ideally this matches what's in
  the Profile, but if we can't confidently split the name into
  first/last we'll just have this.")
   (s/optional-key :email) EMailMap
   (s/optional-key :profile) PendingProfile})

(s/defschema PendingUser
  "User record for a user without an authentication. Might be duplicates."
  (merge PendingUserInput
         RethinkFields))

(s/defschema SignupType
  (s/enum :signup :pending :facebook :default))

(s/defschema CreateInput
  "Schema for the input data passed to create! (depending on the SignupType)."
  (s/either SignupFields FullUserInput PendingUserInput))

(s/defschema User
  (s/either FullUser PendingUser))

;; This one's interesting, because a race might require that they have
;; a bunch of info about that user.
;;
;; This might also be a GUEST user that signed up for a race without
;; an account.


(s/defschema Merge
  "Created in case of a merge. The original user gets destroyed."
  {:id s/Str
   :from s/Str
   :to s/Str})

(s/defn toggle-optional :- {s/Any s/Any}
  "Takes in a Schema, and a keyword to toggle."
  [schema :- {s/Any s/Any}
   k :- (s/named s/Any "Key to toggle")]
  (if-let [optional-v (get schema (s/optional-key k))]
    ;;change to required:
    (-> (assoc schema k optional-v)
        (dissoc (s/optional-key k)))
    ;;change to optional:
    (-> (assoc schema (s/optional-key k) (get schema k))
        (dissoc schema k))))
