(ns hub.user.client.util
  "Client Helpers"
  (:require [clojure.string :as str]
            [hub.queue.client :as qc]
            [hub.user.api.schema :as as]
            [schema.core :as s]))

;; ## Helpers
(s/defn profile :- (s/maybe as/Profile)
  "If the supplied user's valid, returns the profile."
  [user-doc :- as/User]
  (:profile user-doc))

(s/defn email :- (s/maybe s/Str)
  "If the supplied user's valid, returns the email address."
  [user-doc :- as/User]
  (-> user-doc :email :address))

(s/defn display-name :- (s/maybe (s/either as/FullName as/UserName))
  "Takes in a user-doc or username and returns the user's full name,
    or username if the name hasn't been input yet."
  [user-doc :- as/User]
  (or (:name user-doc)
      (when-let [{:keys [first last]} (-> user-doc profile :name)]
        (format "%s %s" first last))
      (:username user-doc)))

(s/defn get-dob-str :- (s/maybe s/Str)
  "Returns date of birth as a string mm/dd/yyyy"
  [user-doc :- as/User]
  (when-let [birthdate (-> user-doc :profile :birthdate)]
    (->> (map birthdate [:month :day :year])
         (str/join "/"))))

(s/defn super-admin? :- s/Bool
  [user-doc :- as/User]
  (contains? (:roles (:username user-doc)) "superadmin"))

(defn publish!
  "Publishes the given topic / payload and returns the payload."
  [topic m]
  (try (qc/publish! topic m)
       ;;TODO: Log error.
       (catch Exception e nil))
  m)
