(ns hub.user.facebook
  "Facebook API helpers."
  (:require [clojure.core.async :as a]
            [clojure.string :as str]
            [environ.core :as e]
            [org.httpkit.client :as http]
            [hub.user.api.schema :as as]
            [hub.user.config :as c]
            [hub.user.oauth :as oauth]
            [hub.user.schema :as us]
            [hub.user.service :as service]
            [hub.user.transforms :as t]
            [hub.util.aws :as aws]
            [hub.util.facebook :as fb]
            [hub.util.api :as ua]
            [schema.core :as s]
            [taoensso.timbre :as log]))

;; ## Api Calls

;; ### Profile Info

(s/defschema CoverPhoto
  {:id (s/named s/Str "ID of the photo in Facebook.")
   (s/optional-key :offset_y) s/Int
   (s/optional-key :source) (s/named s/Str "Photo URL.")})

(s/defschema ProfilePhoto
  {:is_silhouette s/Bool
   :url (s/named s/Str "URL of the photo on Facebook's CDN.")
   (s/optional-key :width) s/Int
   (s/optional-key :height) s/Int})

(s/defn cover-photo :- CoverPhoto
  "Returns information for the supplied user's cover photo. If the
  user's cover photo isn't set, only the :id is present."
  [token :- s/Str]
  (:cover (fb/me token ["cover"])))

(s/defn largest-cover
  "Takes in a FB cover photo and queries the API for the highest
  resolution version of that photo."
  [cover :- CoverPhoto]
  (let [{:keys [height width source]}
        (->> (fb/api-get (:id cover)
                         {:query-params {:type "large" :redirect false}
                          :params {:oauth-token (fb/server-secret)}})
             :images
             (sort-by (complement :width))
             (first))]
    {:url source
     :y-offset (:offset_y cover)
     :height height
     :width width}))

(s/defschema ProfilePhotoOpts
  "Options for the profile-photo API call."
  {:user-id s/Str
   (s/optional-key :width) s/Int
   (s/optional-key :height) s/Int})

(s/defn profile-photo :- (s/maybe ProfilePhoto)
  "If the supplied user exists, returns information for the supplied
  user's cover photo. If the user's cover photo isn't set, only
  the :id is present."
  [opts :- ProfilePhotoOpts]
  (:data
   (fb/api-get (str (:user-id opts) "/picture")
               {:query-params (merge (select-keys opts [:width :height])
                                     {:type "large" :redirect false})
                :params {:oauth-token (fb/server-secret)}})))

;; ## Photo Processing

(s/defn url->photo :- aws/Image
  "For users, the `photo-name` passed to AWS has been main-photo or
  logo."
  [url :- s/Str]
  (let [{:keys [status headers body error]} @(http/get url {:as :stream})]
    (-> (select-keys headers [:content-type])
        (assoc :tempfile body
               :size (Long/parseLong (:content-length headers))))))

;; For users, the `photo-name` passed to AWS has been main-photo or
;;  logo.
(s/defn facebook->aws :- {:success s/Bool
                          :url s/Str}
  "Attempts to transfer the supplied facebook image to AWS. Returns
    a map with :success true if the transfer was successful, false
    otherwise. Either way the result will have the proper URL to
    store.

   If the skip-aws env variable is set, skips upload."
  [facebook-url :- s/Str
   image-type :- (s/enum "main-photo" "logo")]
  (if (not-empty (e/env :skip-aws))
    {:success true :url facebook-url}
    (let [conf (aws/aws-config)
          cloudfront (-> conf :buckets :photos :cloudfront)
          photo (url->photo facebook-url)
          chan (aws/upload-image conf :photos image-type photo)
          {:keys [success url]} (a/<!! chan)]
      {:success success
       :url (if success
              (str cloudfront "/" url)
              facebook-url)})))

;; # Service Methods

(s/defn parse-birthdate :- us/BirthDate
  "Grabs the birthdate object from Facebook's birthday field."
  [fb-birthdate :- s/Str]
  (let [parse-long (fn [x] (if (number? x) x (Long/parseLong x)))
        [month day year] (str/split fb-birthdate #"/")]
    {:month (parse-long month)
     :day (parse-long day)
     :year (parse-long year)}))

(s/defn parse-photos :- (s/maybe {us/ImageType us/ImageDetails})
  "Parses the photos returned by Facebook's API."
  [profile :- ProfilePhoto
   cover :- CoverPhoto]
  (letfn [(aws-xfer [image-type]
            (fn [url] (:url (facebook->aws url image-type))))]
    (not-empty
     (merge (when-not (:is_silhouette profile)
              {:profile (-> (select-keys profile [:url :width :height])
                            (update :url (aws-xfer "logo")))})
            (when-let [source (:source cover)]
              {:cover (-> (largest-cover cover)
                          (update :url (aws-xfer "main-photo")))})))))

(s/defn generate-profile :- us/Profile
  [profile-photo :- ProfilePhoto
   me-response :- fb/UserData]
  (let [{:keys [first_name last_name birthday cover gender]} me-response]
    (merge
     {:name {:first first_name, :last last_name}}
     (when-let [bd (and birthday (parse-birthdate birthday))]
       {:birthdate bd})
     (when-let [photos (parse-photos profile-photo cover)]
       {:photos photos})
     (when-let [gender (#{"male" "female"} gender)]
       {:gender gender}))))

(def racehub-fb-fields
    "Fields we need to ask for when we query Facebook."
    ["birthday" "cover"
     "first_name" "last_name" "name"
     "email" "gender" "verified"])

(s/defschema FacebookMeta
  "Facebook Metadata we store in RH."
  {:name s/Str
   :email s/Str
   :id s/Str})

(s/defmethod oauth/oauth-metadata :facebook :- FacebookMeta
  [auth-type :- oauth/OAuthProvider token :- s/Str]
  (-> (fb/me token)
      (select-keys [:name :email :id])))

(s/defmethod oauth/oauth-name :facebook :- s/Str
  [auth-type :- oauth/OAuthProvider
   metadata :- FacebookMeta]
  (:name metadata))

(s/defmethod service/generate-user :facebook
  [_ {token :access-token id :facebook-id :as m}]
  (let [token (future (:token (fb/exchange-token token)))
        prof-photo (future (profile-photo {:user-id id, :width 400, :height 400}))
        me (future (fb/me @token ["birthday" "cover"
                                  "first_name" "last_name" "name"
                                  "email" "gender" "verified"]))
        {:keys [error email first_name last_name verified]} @me]
    (if (not-empty error)
      (do
        (log/errorf "Error getting Facebook me: %s" error)
        nil)
      {:username (service/generate-username {:email email
                                             :first-name first_name
                                             :last-name last_name})
       :email {:address email
               :verified? verified}
       :profile (generate-profile @prof-photo @me)
       :oauth {:facebook {:token @token
                          :id id
                          :metadata (select-keys @me [:name :email :id])}}})))

(s/defn upgrade-token! :- us/User
    "Takes an existing user and a client side token and attempts to
    upgrade that token to a new, server side token using the Facebook
    API."
    [user :- us/User access-token :- s/Str]
    (if-let [upgraded (:token (fb/exchange-token access-token))]
      (service/update! user {:oauth {:facebook {:token upgraded}}})
      user))

(s/defn merge-facebook :- us/User
  "Merges the generated Facebook user into the current user doc. The
    rule is that keys that already exist in the user's profile (Except
    for the new facebook token) win."
  [user :- us/User facebook-user :- us/FullUserInput]
  (letfn [(merge* [l r]
            (if (map? l)
              (merge-with merge* l r)
              l))]
    (merge-with merge* user facebook-user)))

(s/defschema FBResponse
  (s/either
   {:ok (s/pred true?)
    :user s/Any
    :created? s/Bool}
   {:ok (s/pred false?)
    (s/optional-key :reason) s/Str}))

(s/defn attach-facebook! :- FBResponse
  [user :- us/FullUser m :- fb/AuthMap]
  (if-let [token (:token (fb/exchange-token (:access-token m)))]
    (let [updated (-> (merge-facebook user (service/generate-user :facebook m))
                      (oauth/register-auth! :facebook token (:facebook-id m)))]
      {:ok true :user updated :created false})
    {:ok false :reason "Token exchange failed."}))

(s/defn get-or-create-via-facebook! :- FBResponse
  "If the user exists, updates its facebook token and returns the
    updated user. If the user does NOT exist, creates the user, sends
    a welcome email and returns that."
  [{:keys [facebook-id access-token] :as m} :- fb/AuthMap]
  (or (when-let [existing (service/get-user-by-fb-id facebook-id)]
        {:ok true
         :created? false
         :user (upgrade-token! existing access-token)})
      (if-let [fb-input (service/generate-user :facebook m)]
        (if-let [other (:fulluser
                        (service/get-user-by-email
                         (-> fb-input :email :address)))]
          (if (oauth/oauth-registered? other :facebook)
            {:ok false, :reason "Other user already has a linked FB."}
            {:ok true
             :created? false
             :user (service/put! (merge-facebook other fb-input))})
          (let [new-user (service/create! fb-input)]
            {:ok true, :created? true, :user new-user}))
        {:ok false
         :reason "Problem generating user from facebook."})))

(s/defn attempt :- (s/either as/FullUser ua/Err)
  "Helps with generating API responses off of FBResponse
  objects. Returns an API user."
  [resp :- FBResponse error-reason :- s/Str]
  (if (:ok resp)
    (t/fulluser-model->fulluser-api (:user resp))
    {:error {:type error-reason, :message (:reason resp)}}))
