(ns hub.user.setup
  "Setup of user service - create database, tables, indices."
  (:require  [hub.util.rethink :as ur :refer [ args]]
             [rethinkdb.query :as r]
             [schema.core :as s]))

(s/defn setup!
  "Basic setup for the user table."
  [db-name :- s/Str]
  (let [maybe-index (fn [table-name index-name query-fn]
                      (ur/maybe-create-index
                       index-name
                       (r/table (r/db db-name) table-name)
                       query-fn))
        full-index (partial maybe-index "fulluser")
        pending-index (partial maybe-index "pendinguser")]
    ;;create racehub database, if needed
    (ur/run (ur/try-create db-name))

    ;;create full user table, if needed
    (ur/run (ur/try-table db-name "fulluser"))

    ;;create pending user table, if needed
    (ur/run (ur/try-table db-name "pendinguser"))

    ;;FullUser Index: Username
    (full-index "username" (r/fn [row]
                             (r/downcase
                              (r/get-field row :username))))

    ;;FullUser Index: Facebook ID
    (full-index "facebook-id"
                (r/fn [row]
                  (r/downcase
                   (ur/get-field-in row [:oauth :facebook :metadata :id]))))

    ;;FullUser Index: Email Address
    (full-index "email" (r/fn [row]
                          (r/downcase
                           (ur/get-field-in row [:email :address]))))

    ;;PendingUser Index: Email Address
    (pending-index "email"
                   (r/fn [row]
                     (r/downcase
                      (ur/get-field-in row  [:email :address]))))

    (full-index "name"
                (r/fn [row]
                  (-> (r/add (ur/get-field-in row [:profile :name :first])
                             (ur/get-field-in row [:profile :name :last]))
                      (r/split)
                      args
                      r/add
                      r/downcase)))

    ;;PendingUser Index: Name
    (pending-index "name"
                   (r/fn [row]
                     (r/add (args (r/split (r/downcase (r/get-field row :name)))))))

    ;;FullUser Index: Reset Code
    (full-index "pw-reset-code"
                (r/fn [row]
                  (ur/get-field-in row [:password-reset :code])))

    ;;FullUser Index: Facebook Email
    (full-index "facebook-email"
                (r/fn [row]
                  (r/downcase
                   (ur/get-field-in row [:oauth :facebook :metadata :email]))))

    ;;FullUser Index: Facebook ID
    (full-index "facebook-id"
                (r/fn [row]
                  (r/downcase
                   (ur/get-field-in row [:oauth :facebook :metadata :id]))))))
