(ns hub.util.facebook
  "Facebook API helpers."
  (:require [clojure.string :as str]
            [cheshire.core :refer [parse-string]]
            [environ.core :as e :refer [env]]
            [clj-http.client :as client]
            [org.httpkit.client :as http]
            [ring.util.codec :as ring-codec]
            [taoensso.timbre :as log]
            [schema.core :as s]))

;; This namespace requires FB_APP_SECRET, FB_CLIENT_ID, and
;; FB_CLIENT_SECRET environment variables.

;; ## Schema

(s/defschema CallBack
  {:path s/Str
   :domain s/Str})

(s/defschema TokenLocation
  (s/enum :params :body))

(s/defschema OAuthConfig
  {:token-location TokenLocation
   :auth-url s/Str
   :token-url s/Str
   :client-id s/Str
   :client-secret s/Str
   (s/optional-key :auth-query) {s/Keyword s/Str}
   (s/optional-key :callback) CallBack})

(s/defschema APIError
  "Error response from the Facebook API."
  {:error {:message s/Str :type s/Str :code s/Int}})

(s/defschema AuthMap
  {:facebook-id s/Str
   :access-token s/Str})

(s/defschema FacebookConfig
  {:oauth OAuthConfig
   :app-secret s/Str})

;; ## Facebook API Data

(def api-version "2.9")
(def root (str "https://graph.facebook.com/v" api-version "/"))

(s/defn facebook-config :- FacebookConfig
  "Token location specifies that the token is going to come back in
  the params, not the body. We also make sure to ask for email
  privileges, to beef up a particular user's profile.
  The returned map contains oauth information AND the app-specific
  secret key for facebook."
  []
  {:app-secret (env :fb-app-secret)
   :oauth {:token-location :params
           :auth-url "https://www.facebook.com/dialog/oauth"
           :token-url "https://graph.facebook.com/oauth/access_token"
           :auth-query {:scope "email", :response_type "code"}
           :client-id (env :fb-client-id)
           :client-secret (env :fb-client-secret)}})

(def app-id (comp :client-id :oauth facebook-config))
(def client-secret (comp :client-secret :oauth facebook-config))
(def server-secret (comp :app-secret facebook-config))

(s/defn body
  "returns a json-parsed representation of the body of the response."
  [req]
  (parse-string (:body @req) keyword))

(defn get-access-token-from-params
  "Alternate function to allow retrieve
  access_token when passed in as form params."
  [{body :body}]
  (when body
    (-> body parse-string (get "access_token"))))

(s/defn access-token :- (s/maybe s/Str)
  "Generates a server-side access token. This is used to make calls
  about the app to the server.
  More info:
  https://developers.facebook.com/docs/facebook-login/access-tokens"
  []
  (let [params {:client_id (app-id)
                :client_secret (client-secret)
                :grant_type "client_credentials"}]
    (get-access-token-from-params
     @(http/get (str root "oauth/access_token")
                {:query-params params}))))

(s/defschema GetOpts
  {(s/optional-key :query-params) {s/Any s/Any}
   (s/optional-key :params) {s/Any s/Any}})

(s/defn api-get :- {s/Any s/Any}
  "Triggers a get request to the server."
  [endpoint :- s/Str
   {:keys [query-params params]} :- GetOpts]
  (let [opts (assoc params :query-params query-params)]
    (body (http/get (str root endpoint) opts))))

(s/defn api-post
  "Triggers a post request to the server."
  [endpoint :- s/Str
   {:keys [query-params params]} :- GetOpts]
  (let [opts (assoc params :query-params query-params)]
    (body (http/post (str root endpoint) opts))))

(s/defn api-delete
  "Triggers a delete request to the server."
  [endpoint :- s/Str
   {:keys [query-params params]} :- GetOpts]
  (let [opts (assoc params :query-params query-params)]
    (body (http/delete (str root endpoint) opts))))

;; ## Api Calls

;; ### Token Exchange
;;
;; This section deals with exchanging short-lived client tokens for
;;long-lived server tokens.

(s/defn exchange-token :- (s/either APIError {:token s/Str})
  "Accepts a client-side short-term token from the Facebook JS SDK and
  exchanges it for a long-lived token. The long token lasts for about
  60 days from last use.  If the user logs in with facebook, or uses
  our site and our JS SDK refreshes them, the token renews for another
  sixty days.  The client side token by itself is only good for an
  hour or so."
  ;;https://developers.facebook.com/docs/facebook-login/access-tokens#usertokens
  [token :- s/Str]
  (let [{:keys [client-id client-secret]} (:oauth (facebook-config))
        resp (client/get (str root "oauth/access_token")
                         {:query-params
                          {:grant_type "fb_exchange_token"
                           :fb_exchange_token token
                           :client_id client-id
                           :client_secret client-secret}})]
    (log/infof "Exchange Token: %s" token)
    (log/infof "Exchange Token Response: %s" (:body resp))
    (or (when-let [token (get-access-token-from-params resp)]
          {:token token})
        (parse-string (:body resp) keyword)
        (do
          (log/errorf "Exchange Token Response Nil!. Response: %s" resp)
          {:error {:message "Nil response."
                   :type "Nil"
                   :code "Nil"}}))))

;; ### Profile Info

(s/defschema UserData
  "Response from Facebook's me endpoint. The keys here are really just
  examples of the fields that can come through. If you ask for
  specific fields, the rest will be missing, and certain users will
  just not have certain fields."
  {(s/optional-key :email) s/Str
   (s/optional-key :first_name) s/Str
   (s/optional-key :timezone) (s/named s/Int "UTC offset.")
   (s/optional-key :locale) s/Str
   (s/optional-key :name) s/Str
   (s/optional-key :updated_time) s/Str
   (s/optional-key :link) s/Str
   (s/optional-key :id) s/Str
   (s/optional-key :last_name) s/Str
   (s/optional-key :gender) s/Str
   (s/optional-key :verified) s/Bool
   s/Any s/Any})

(def racehub-fb-fields
    "Fields we need to ask for when we query Facebook."
    ["birthday" "cover"
     "first_name" "last_name" "name"
     "email" "gender" "verified"])

(s/defschema Account
  {:access_token s/Str
   :category s/Str ;;company
   :name s/Str     ;;Paddleguru
   :id s/Str
   :perms [s/Str]
   s/Any s/Any})

(s/defn my-accounts :- [Account]
  "Doesn't support paging right now. Returns all the Accounts for the
  given user token."
  ([token :- s/Str]
   (:data (api-get "me/accounts" {:query-params {:access_token token}}))))

(s/defschema Permission
  {:permission s/Str
   :status (s/enum "granted" "declined")})

(s/defn my-permissions :- [Permission]
  "Doesn't support paging right now. Returns all the Accounts for the
  given user token."
  ([token :- s/Str]
   (:data (api-get "me/permissions" {:query-params {:access_token token}}))))

(s/defn me :- (s/either UserData APIError)
  "Makes a call to Facebook's /user/id endpoint for the user linked to
  the supplied token. Gives you a map with a bunch of information
  about the user.
  Optionally you can supply a list of fields that you want
  returned. If you don't supply any, you get a grab-bag of fields.
  See the documentation for more details on what the map holds:
  https://developers.facebook.com/docs/graph-api/reference/v2.1/user"
  ([token :- s/Str]
   (me token nil))
  ([token :- s/Str fields :- [s/Str]]
   (api-get "me" {:query-params {:access_token token
                                 :fields (str/join "," fields)}})))
