(ns analyze.examples.ibdknox
  (:import clojure.lang.LineNumberingPushbackReader
           java.io.StringReader
           java.io.BufferedReader)
  (:require [clojure.string :as string]
            [analyze.core :as analyze]))

(defn lined-read [string]
  (let [rdr (LineNumberingPushbackReader. (StringReader. string))]
    (reduce conj
            []
            (take-while identity (repeatedly #(read rdr false nil))))))

(defn ->lines [string]
  (vec (line-seq (BufferedReader. (StringReader. string)))))

(defn ->block [lines]
  (clojure.string/join "\n" lines))

(defn line-range [lines start end]
  (when (and start end)
    (map lines (range (dec start) end))))

(defmulti handle-node :op)
(defmethod handle-node :def [cur]
  (let [m (meta (:var cur))]
    {:type :def
     :symbol (:name m)
     :ns (-> (:ns m)
             str
             symbol)
     :val (:init cur)}))

(defmethod handle-node :invoke [cur]
  )

(defmethod handle-node :default [_]
  _)

(defn cat [func expr]
  (apply concat
    (when-let [res (func expr)] [res])
    (map (partial cat func) (:children expr))))

(defn ops [expr]
  (when-let [o (:op expr)]
    o))

(defn by-op-type [op]
  (fn [expr]
    (when (= op (:op expr))
      expr)))

(defn var-invokes [tree & [filt]]
  (let [f (or filt identity)]
    (set (filter f (for [v (cat (by-op-type :invoke) tree)
                         :let [ivar (-> v :fexpr :var)
                               env (-> v :env)
                               m (-> ivar meta)]
                         :when ivar]
                     {;:var ivar
                      :ns (symbol (str (:ns m)))
                      :name (:name m)
                      :col (:column env)
                      :line (dec (:line env))}
                     )))))

(defn all-vars [tree & [filt]]
  (let [f (or filt identity)]
    (set (filter f (for [v (cat (by-op-type :var) tree)
                         :let [ivar (-> v :var)
                               env (-> v :meta)
                               m (-> ivar meta)]
                         :when ivar]
                     {;:var ivar
                      :ns (symbol (str (:ns m)))
                      :name (:name m)
                      :col (:column env)
                      :line (if (:line env)
                              (dec (:line env))
                              -1)}
                     )))))

(defn form-name [tree]
  (when (= (:op tree) :def)
    (let [m (meta (:var tree))]
      {:name (:name m)
       :ns (symbol (str (:ns m)))})))

(defn not-core [ivar]
  (not= 'clojure.core (:ns ivar)))

(defn handle-form [lines nsp form]
  (when (seq form)
    (let [{:keys [line endline column]} (meta form)
          text (->block (line-range lines line endline))
          ana (try
                (analyze/analyze-one {:ns {:name nsp} :context :eval} form)
                (catch Exception e
                  (println "BROKEN FORM: " nsp " :: " (pr-str form))
                  (.printStackTrace e)
                  (println "Error: " (.getMessage e))))]
    {:form form
     :line line
     :column column
     :text text
     :name (form-name ana)
     :vars (all-vars ana not-core)
     :invokes (var-invokes ana not-core)}
      )))

(defn ->analyzed [f nsp]
  (require nsp)
  (let [contents (slurp f)
        forms (lined-read contents)
        lines (->lines contents)]
    (doall (for [cur-form forms]
             (handle-form lines nsp cur-form)))))

;(analyze.core/analyze-one {:ns {:name 'lightable.server} :context :eval} '(ns lightable.server))

;(->analyzed "/users/chris/repos/lightable/src/lightable/server.clj" 'lightable.server)

(comment
(let [contents (slurp "examplefile.clj")
      forms (lined-read contents)
      lines (->lines contents)]
  (first
  (for [f forms
        :let [{:keys [line endline]} (meta f)
              text (->block (line-range lines line endline))]]
    {:form f
     :text text
     :analysis (analyze/analyze-one {:ns {:name 'clojure.repl} :context :eval} f)})))


  )
