(ns ring-aws-api-gateway-lambda-adapter.core
  (:require [clojure.string :as str]
            [clojure.java.io :as io]
            [clojure.data.json :as json]
            [uswitch.lambada.core :refer [deflambdafn]])
  (:import java.lang.Integer
           java.lang.NumberFormatException))

(defn maybe-encode-json [json]
  (try
    (json/write-str json)
      (catch Exception e
        json)))

(defn event->request
  "Transform lambda input of an API Proxy request to Ring requests.
   Has two extra properties:
   :event - the lambda input
   :context - an instance of a lambda context
              http://docs.aws.amazon.com/lambda/latest/dg/java-context-object.html"
  [event context]
  (let [[http-version host]
        (str/split (get-in event [:headers :Via] "") #" ")]
    {:server-port
     (try
       (Integer/parseInt (get-in event [:headers :X-Forwarded-Port]))
       (catch NumberFormatException e nil))
     :body           (get event :body)
     :server-name    host
     :uri            (get event :path "/")
     :query-params   (get event :queryStringParameters {})
     :scheme         (keyword
                      (get-in event [:headers :X-Forwarded-Proto]))
     :request-method (keyword
                      (str/lower-case (get event :httpMethod "")))
     :protocol       (format "HTTP/%s" http-version)
     :headers        (into {} (map (fn -header-keys [[k v]]
                                     [(str/lower-case (name k)) v])
                                   (:headers event)))
     :event          event
     :context        context}))

(defn handle-request
  "Handle a lambda invocation as a ring request. Writes ring response as JSON to
  `out` for 200 responses, raises an exception with ring response as JSON for
  non-200 responses"
  [handler in out context]
  (let [event (json/read (io/reader in)
                         :key-fn keyword)
        request (event->request event context)
        response (handler request)
        status (:status response)
        body   (:body response)
        formatted (-> response
                      (assoc :statusCode status)
                      (assoc :body (maybe-encode-json body))
                      (select-keys [:statusCode :headers :body]))]
    (with-open [w (io/writer out)]
      (json/write formatted w))))

(defmacro defhandler
  "A ring handler for AWS Lambda and AWS API Gateway"
  [name handler]
  `(deflambdafn ~name
     [in# out# context#]
     (handle-request ~handler in# out# context#)))
