;;; Copyright (c) Rachel Bowyer 2022, 2023. All rights reserved.
;;;
;;; This program and the accompanying materials
;;; are made available under the terms of the Eclipse Public License v1.0
;;; which accompanies this distribution, and is available at
;;; http://www.eclipse.org/legal/epl-v10.html
;;;
;;; The functions "file-and-line" and "assert-predicate" are adapted from
;;; functions that are marked as "Copyright (c) Rich Hickey. All rights reserved."
;;; and distributed as part of clojure.test. These functions are used under the
;;; terms of the EPL v 1.0.

(ns info.bowyer.sam-diff-humane.core
  (:require
    [clojure.pprint :as pp]
    [clojure.string :as str]
    [clojure.test :refer [assert-expr do-report with-test-out testing-vars-str]]
    [info.bowyer.sam-diff.core :as sam-diff]))


;;;; Report differences

;; todo  long form diffs

(defn- file-and-line
  []
  (#'clojure.test/stacktrace-file-and-line (drop-while
                              #(let [cl-name (.getClassName ^StackTraceElement %)]
                                 (or (str/starts-with? cl-name "sam_diff.")
                                     (str/starts-with? cl-name "java.lang.")
                                     (str/starts-with? cl-name "clojure.test$")
                                     (str/starts-with? cl-name "clojure.core$ex_info")))
                              (.getStackTrace (Thread/currentThread)))))

(defn show-diffs
  [expected actuals a-v r-v]
  (with-test-out
    (println "Equality Test failed")
    (let [m         (file-and-line)
          test-name (testing-vars-str m)]
      (println test-name ))

    (println)
    (println expected)
    (pp/pprint a-v)

    (doseq [[b-v actual] (map vector r-v actuals)]
      (println)
      (println actual)
      (pp/pprint b-v))

    (println)
    (println "Diffs")
    (println)

    (doseq [b-v r-v]
      (println (sam-diff/diff a-v b-v)))))


;;;; Code from clojure.test (but modified)

(defn- assert-predicate
  "Returns generic assertion code for any functional predicate.  The
  'expected' argument to 'report' will contains the original form, the
  'actual' argument will contain the form with all its sub-forms
   evaluated.  If the predicate returns false, the 'actual' form will
   be wrapped in (not...)."
  {:added "1.1"}
  [msg form]
  (let [args (rest form)
        pred (first form)
        expected (str (first args))
        actuals (map str (rest args))]
    `(let [values# (list ~@args)
           a-v# (first values#)
           r-v# (rest values#)
           result# (apply ~pred values#)]
       (when (empty? r-v#)
         (throw (Exception. "= expects more than one argument")))
       (when-not result#
         (show-diffs ~expected (list ~@actuals) a-v# r-v#))
       (if result#
         (do-report {:type     :pass, :message ~msg,
                     :expected '~form, :actual (cons '~pred values#)})
         (do-report {:type :fail, :message ~msg,
                     :expected '~form, :actual (list '~'not (cons '~pred values#))}))
       result#)))


;;;; Hook Clojure test

(defonce activation-body
         (delay
           (defmethod assert-expr '= [msg form]
             (assert-predicate msg form))
           (defmethod assert-expr 'clojure.core/= [msg form]
             (assert-predicate msg form))))

(defn activate! []
  @activation-body)