(ns sam-diff.formatter.string
  (:require [clojure.string :as str]
            [sam-diff.colours :as colours]
            [clojure.pprint :as pp]))

;; todo - might need a more sophisticated version of consolidate to work with
;; the newer algos
;; In particular if there is an [ins] and a [del] at the same point then move
;; the [ins] to come after the [del]. It just makes the output look nicer.


(defn- consolidate
  "Consolidates consecutive inserts or deletes"
  [{:keys [inserted deleted a] :as diff-info}]
  (letfn [(consolidate-one [t data]
            ;(println "here")
            (if data
              (reduce
                (fn [acc e]
                  ;(println "e" e)
                  (cond-> acc
                          (and (acc e) (not (sequential? (acc e))))
                          (update e list)

                          ;; Consolidate either if deleting
                          ;; or if inserting and no non-deleted char
                          (and (acc (dec e))
                               (or (= t :del)
                                   (contains? deleted (dec e))))
                          (-> (update (dec e) #(concat (acc e)
                                                       (cond-> % (not (sequential? %)) list)))
                              (dissoc e))))
                  data
                  (range (-> a count) -1 -1))

              {}))]
    (-> diff-info
        (assoc :consolidated-inserted (consolidate-one :ins inserted))
        (assoc :consolidated-deleted (consolidate-one :del deleted)))))

(defn- layout
  "Assembles the final string"
  [{:keys [a deleted consolidated-inserted consolidated-deleted] :as _diff-info}]
  (let [a-lookup
        (->> a (map-indexed (fn [idx e] [(- (count a) idx 1) e])) (into {}))]
    (loop [i      0
           result []]
      (if (> i (count a))
        result
        (let [next-result
              (let [ins (consolidated-inserted i)]
                (cond->> result
                         ins
                         (cons {:ins ins})))


              next-result
              (let [del (consolidated-deleted i)]
                (cond->> next-result
                  del
                  (cons {:del del})

                  (and (or (nil? deleted) (not (deleted i)))
                       (< i (count a)))
                  (cons (a-lookup i))))]
          (recur (inc i) next-result))))))

(defn- render
  "Renders the laid out diffs as a string"
  [laid-out-str {:keys [format] :as _opts}]
  (->> laid-out-str
       (reduce
         (fn [bob e]
           (cond
             (:del e)
             (if (= format :colour)
               (.append bob (str colours/red "[" (str/join (flatten (:del e))) "]" colours/reset))
               (.append bob (str "[del " (str/join (flatten (:del e))) "]")))

             (:ins e)
             (if (= format :colour)
               (.append bob (str colours/blue  "[" (str/join (flatten (:ins e))) "]" colours/reset))
               (.append bob (str "[ins " (str/join (flatten (:ins e))) "]")))

             (char? e)
             (.append bob e)))
         (StringBuilder.))
       str))

(defn frmt
  [{:keys [a] :as diff-info} opts]
  (str
    (when (> (count a) 1000)
      (str (with-out-str (pp/pprint (dissoc diff-info :a))) "\n"))
    (-> diff-info
        consolidate
        layout
        (render opts))))



