(ns sam-diff.levenshtein.cost-of
  (:require [sam-diff.levenshtein.classify :as classify]))

(def cost-of-nil 0)
(def cost-of-atom 1)


(declare cost-of-with-cache)

(defn- cost-of-string
  [v]
  (count v))


(defmulti cost-of-impl (fn [_cache x] (classify/classify x)))

(defmethod cost-of-impl :nil
  [cache _v]
  [cache cost-of-nil])

(defmethod cost-of-impl :atom
  [cache _v]
  [cache cost-of-atom])

(defmethod cost-of-impl :string
  [cache v]
  [cache (cost-of-string v)])

(defmethod cost-of-impl :number
  [cache v]
  [cache cost-of-atom])

(defmethod cost-of-impl :keyword
  [cache v]
  [cache cost-of-atom])

(defmethod cost-of-impl :symbol
  [cache v]
  [cache cost-of-atom])

(defmethod cost-of-impl :seq
  [cache v]
  (reduce (fn [[acc-cache result] e]
            (update (cost-of-with-cache acc-cache e) 1 + result))
          [cache 0]
          v))

(defmethod cost-of-impl :set
  [cache v]
  (cost-of-impl cache (seq v)))

(defmethod cost-of-impl :map
  [cache v]
  (cost-of-impl cache (map (fn [[_ v]] v) v)))

(defmethod cost-of-impl :vector
  [cache v]
  (cost-of-impl cache (seq v)))


(defn- cost-of-with-cache
  [cache v]
  (if-let [result (get cache v)]
    [cache result]
    (let [[cache result]  (cost-of-impl cache v)
          new-cache       (assoc cache v result)]
      [new-cache result])))

(defn cost-of
  [v]
  (let [[_cache result] (cost-of-with-cache {} v)]
    result))