;; This namespace defines interval trees and functions
;; for working with them.  The interval trees are implemented 
;; as augmented red-black trees following the approach laid out in 
;; Introduction to Algorithms (2009) by Cormen, Leiserson, Rivest and Stein (CLRS).  Interval trees
;; are binary search trees that stores real valued intervals, for example [3.5, 5.7].  The node keys
;; are the low end of the interval, and each node also contain the 
;; the maximum of the high values of itself and its branches.  This allows
;; for logarithmic time searches of all intervals that contain a point
;; and all interval that overlap a given interval.
(ns interval-tree.core
	(:gen-class)
	(:require [interval-tree.interval]))

;; ## red
;; A symbol representing the color red for red-black tree node annotation.
(def red "red")

;; ## black
;; A symbol representing the color black for red-black tree node annotation.
(def black "black")

;; ## low
;; An alias for interval-tree.interval/low.
(defn low [interval] (interval-tree.interval/low interval))

;; ## high
;; An alias for interval-tree.interval/high.
(defn high [interval] (interval-tree.interval/high interval))

;; ## Interval-tree node constructor
;; Interval tree nodes have two additional pieces of data
;; the interval itself (interval-tree.interval/new low high)
;; and the max endpoint of any interval in its subtree.
;; The key is the low interval value, thus duplicate keys 
;; are permitted.
(defn node 
	([]
		(ref {:key nil 
		      :value nil
		      :color nil 
		      :left nil
		      :right nil
		      :parent nil
		      :interval nil
		      :max nil}))
	([key value color left right parent interval max]
		(let [self (node)]
			(dosync
				(alter self assoc :key key)
				(alter self assoc :value value)
				(alter self assoc :color color)
				(alter self assoc :left left)
				(alter self assoc :right right)
				(alter self assoc :parent parent)
				(alter self assoc :interval interval)
				(alter self assoc :max max))
			self)))

;; ## Node getters and setters
(defn get-key [self] (:key @self))
(defn get-value [self] (:value @self))
(defn get-color [self] (:color @self))
(defn get-left [self] (:left @self))
(defn get-right [self] (:right @self))
(defn get-parent [self] (:parent @self))
(defn get-interval [self] (:interval @self))
(defn get-max [self] (:max @self))
(defn set-key [self key] (dosync (alter self assoc :key key)))
(defn set-value [self value] (dosync (alter self assoc :value value)))
(defn set-color [self color] (dosync (alter self assoc :color color)))
(defn set-left [self left] (dosync (alter self assoc :left left)))
(defn set-right [self right] (dosync (alter self assoc :right right)))
(defn set-parent [self parent] (dosync (alter self assoc :parent parent)))
(defn set-interval [self interval] (dosync (alter self assoc :interval interval)))
(defn set-max [self max] (dosync (alter self assoc :max max)))

;; ## Interval-tree constructor
;; The sentinel is the shared node that acts as branch gaurdian
;; root is the root of the tree.
(defn new []
	(let [sentinel (node nil nil black nil nil nil nil nil)
	      root (node nil nil black sentinel sentinel sentinel nil nil)
	      self (ref {:sentinel sentinel :root root})]
		self))

;; ## Tree getters and setters
(defn get-sentinel [self] (:sentinel @self))
(defn get-root [self] (:root @self))
(defn set-root [self node] (dosync (alter self assoc :root node)))

;; ## pretty-print
;; This function pretty prints an interval tree.  For each node
;; the intervals low and high value are printed in addition
;; to the maximum high value of the node itself and its branches.
(defn pretty-print 
	([tree]
		(if (not= (get-key (get-root tree)) nil) 
			(pretty-print (get-root tree) 0 (get-sentinel tree))))
	([node indent sentinel]
		(let [indent-string (apply str (repeat indent " ")) interval (get-interval node)]
			(if (= node sentinel)
				(println (str indent-string "sentinel"))
				(do
					(if (= (get-key (get-parent node)) nil)
						(println 
							(str indent-string (get-key node) " : " (get-color node) 
						     	" [interval " (low interval) " " (high interval) "]"
								" [max " (get-max node) "]"))
						(println 
							(str indent-string (get-key node) " : " (get-color node) 
						     	" [parent-key " (get-key (get-parent node)) "]" 
						     	" [interval " (low interval) " " (high interval) "]"
								" [max " (get-max node) "]")))
					(pretty-print (get-left node) (+ indent 2) sentinel)
					(pretty-print (get-right node) (+ indent 2) sentinel))))))

;; ## max-of-three
;; Given three numerical, some possibly nil values, 
;; this function returns the largest of the three. This function
;; is used during tree rebalancing to update the max child high value.
(defn max-of-three [element-one element-two element-three]
	(apply max
		(remove nil? 
			[element-one element-two element-three])))

;; ## recursive-max
;; A function to explicitly calculate the maximum high 
;; value of a node's child branches.
(defn recursive-max [tree node]
	(if (= node (get-sentinel tree))
		nil
		(max-of-three
			(high (get-interval node))
			(recursive-max tree (get-left node))
			(recursive-max tree (get-right node))))) 

;; ## check-max-interval
;; This function is used in testing to verify that the augmented
;; node max value is correct.
(defn check-max-interval [tree node]
	(if (= node (get-sentinel tree))
		true
		(let [condition-one 
			(= 
				(get-max node) 
				(max-of-three 
					(high (get-interval node))
					(recursive-max tree (get-left node))
					(recursive-max tree (get-right node))))
		      condition-two (check-max-interval tree (get-left node))
		      condition-three (check-max-interval tree (get-right node))]
			(if (= condition-one condition-two condition-three true)
				true
				false))))  	

;; ## check-parent-child-links
;; Takes a tree and verifies that all of the parent child linkages
;; are correct.
(defn check-parent-child-links [tree node]
	(if (= node (get-sentinel tree))
		true
		(let [left (get-left node) 
		      right (get-right node)
			  sentinel (get-sentinel tree)
		      condition-one (if (or (= left sentinel) (= node (get-parent left))) true false)
		      condition-two (if (or (= right sentinel) (= node (get-parent right))) true false)
		      condition-three (check-parent-child-links tree left)
		      condition-four (check-parent-child-links tree right)] 
			(if (and condition-one condition-two condition-three condition-four)
				true
				false)))) 

;; ## check-double-reds
;; Takes a tree and checks whether or not there are double reds.  Returns
;; true if there are no double reds, false if there are.
(defn check-double-reds [tree node]
	(if (= node (get-sentinel tree))
		true
		(let [left (get-left node) right (get-right node) color (get-color node)]
			(if (= color (get-color left) red)
				false
				(if (= color (get-color right) red)
					false
					(if (and (= (check-double-reds tree left) true)
					         (= (check-double-reds tree right) true))
						true
						false))))))

;; ## health-check
;; Takes a tree and verifies that it is correct.
(defn health-check [tree]
	(let [root (get-root tree)]
		(if (and (check-max-interval tree root)
		         (check-parent-child-links tree root)
		         (check-double-reds tree root))
		true
		false)))

;; ## add-to-result
;; This function is called while querying the tree to accumlate 
;; results.  Each result is a vector contain the node's interval
;; and it's value.
(defn add-to-result [node results]
	(dosync 
		(alter results conj [(get-interval node) (get-value node)])))

;; ## vectorize
;; Flattens a tree into a vector of interval value pairs.  In other words,
;; turn the tree's nodes into a list of their contents.
(defn vectorize 
	([tree] 
		(let [results (ref [])]
			(vectorize tree (get-root tree) (get-sentinel tree) results)
			@results))
	([tree node sentinel results]
		(if (not= node sentinel)
			(do 
				(add-to-result node results)
					(vectorize tree (get-left node) sentinel results)
					(vectorize tree (get-right node) sentinel results)))))

;; ## update-max 
;; Updates the nodes max annotation after rotation.
(defn update-max [node]
	(set-max node
		(max-of-three 
			(get-max (get-left node)) 
			(get-max (get-right node)) 
			(high (get-interval node)))))	

;; ## update-max-up-tree
;; Given a tree and a node in the tree, this function
;; climbs up the tree updating the max annotations
;; until no longer necessary.
(defn update-max-up-tree [tree node]
	(dosync
		(let [w (ref node) sentinel (get-sentinel tree)]
			(loop []
				(if (and (not= @w sentinel) (not= @w nil))
					(do
						(update-max @w)
						(ref-set w (get-parent @w))
						(recur)))))))

;; ## left-rotate
;; Rotate node to the left (CLRS pg. 278). 
(defn left-rotate [tree node]
	(let [y (get-right node)]
		(set-right node (get-left y))
		(if (not= (get-left y) (get-sentinel tree))
			(set-parent (get-left y) node))
		(set-parent y (get-parent node))
		(if (= (get-parent node) (get-sentinel tree))
			(set-root tree y)
			(if (= node (get-left (get-parent node)))
				(set-left (get-parent node) y)
				(set-right (get-parent node) y)))
		(set-left y node)
		(set-parent node y)
		(set-max y (get-max node))
		(update-max node)))

;; ## right-rotate 
;; Rotate node to the right (CLRS pg. 278).
(defn right-rotate [tree node]
	(let [y (get-left node)]
		(set-left node (get-right y))
		(if (not= (get-right y) (get-sentinel tree))
			(set-parent (get-right y) node))
		(set-parent y (get-parent node))
		(if (= (get-parent node) (get-sentinel tree))
			(set-root tree y)
			(if (= node (get-right (get-parent node)))
				(set-right (get-parent node) y)
				(set-left (get-parent node) y)))
		(set-right y node)
		(set-parent node y)
		(set-max y (get-max node))
		(update-max node))) 

;; ## insert-fixup 
;; Rebalance the tree after insertion (CLRS pg. 281).
(defn insert-fixup [tree node]
	(let [node-ref (ref node)]
		(while (= (get-color (get-parent @node-ref)) red)
			(if (= (get-parent @node-ref) (get-left (get-parent (get-parent @node-ref))))
				(let [y (get-right (get-parent (get-parent @node-ref)))]
					(if (= (get-color y) red)
						(do (set-color (get-parent @node-ref) black)
							(set-color y black)
							(set-color (get-parent (get-parent @node-ref)) red)
							(ref-set node-ref (get-parent (get-parent @node-ref))))
						(do (if (= @node-ref (get-right (get-parent @node-ref)))
							(do (ref-set node-ref (get-parent @node-ref))
							    (left-rotate tree @node-ref)))
							(set-color (get-parent @node-ref) black)
							(set-color (get-parent (get-parent @node-ref)) red)	
							(right-rotate tree (get-parent (get-parent @node-ref))))))
				(let [y (get-left (get-parent (get-parent @node-ref)))]
					(if (= (get-color y) red)
						(do (set-color (get-parent @node-ref) black)
							(set-color y black)
							(set-color (get-parent (get-parent @node-ref)) red)
							(ref-set node-ref (get-parent (get-parent @node-ref))))
						(do (if (= @node-ref (get-left (get-parent @node-ref)))
							(do (ref-set node-ref (get-parent @node-ref))
							    (right-rotate tree @node-ref)))
							(set-color (get-parent @node-ref) black)
							(set-color (get-parent (get-parent @node-ref)) red)
							(left-rotate tree (get-parent (get-parent @node-ref))))))))
	(set-color (get-root tree) black)
	(update-max (get-root tree))))

;; ## insert
;; Insert and interval value pair (CLRS pg. 280). This is
;; standard binary search tree insertion followed by rebalancing. 
(defn insert [tree interval value]
	(dosync
		(if (= (get-key (get-root tree)) nil)
			(let [root (get-root tree)] 
				(set-key root (low interval))
				(set-value root value)
				(set-interval root interval)
				(set-max root (high interval))
				(set-left root (get-sentinel tree))
				(set-right root (get-sentinel tree)))
			(let [low-key (low interval)
			      high-key (high interval)
			      node (interval-tree.core/node low-key value red nil nil nil interval high-key) 
				  y (ref (get-sentinel tree)) 
				  x (ref (get-root tree))]
				(loop []
					(if (not= @x (get-sentinel tree))
						(do
							(ref-set y @x)
							(if (> (high interval) (get-max @y))
								(set-max @y (high interval)))	
							(if (< (get-key node) (get-key @x))
								(ref-set x (get-left @x))
								(ref-set x (get-right @x)))
							(recur))))
				(do
					(set-parent node @y)
					(if (< (get-key node) (get-key @y))
						(set-left @y node)
						(set-right @y node))
					(set-left node (get-sentinel tree))
					(set-right node (get-sentinel tree))
					(set-color node red)
					(insert-fixup tree node))))))

;; ## lookup-node
;; Given and interval, locates and the numerically equivalent node in the tree
;; if it exists, nil otherwise.
(defn lookup-node [tree interval]
	(dosync
		(if (not= (get-key (get-root tree)) nil)
			(let [node-ref (ref (get-root tree))]
				(loop []
					(if (= @node-ref (get-sentinel tree))
						nil
						(if (and (== (low interval) (low (get-interval @node-ref)))
								 (== (high interval) (high (get-interval @node-ref))))
							@node-ref
							(do 
								(if (< (low interval) (get-key @node-ref))
									(ref-set node-ref (get-left @node-ref))
									(ref-set node-ref (get-right @node-ref)))
								(recur)))))))))

;; ## lookup
;; Takes and interval and returns the value associated with 
;; the numerically equivalent interval in the tree 
;; if it exists, nil otherwise.
(defn lookup [tree interval]
	(let [node (lookup-node tree interval)]
		(if (not= node nil)
			(get-value node))))

;; ## tree-minimum
;; If called with the tree, returns the leftmost node in the tree, else
;; if called with the tree and a node, returns the leftmost child of the node.
(defn tree-minimum 
	([tree]
		(tree-minimum tree (get-root tree)))
	([tree node]
		(let [sentinel (get-sentinel tree)]
			(loop [min-node node]
				(let [left (get-left min-node)]
					(if (= left sentinel)
						min-node
						(recur left)))))))

;; ## tree-maximum
;; If called with the tree, returns the rightmost node in the tree, else
;; if called with the tree and a node, returns the rightmost child of the node.
(defn tree-maximum 
	([tree]
		(tree-maximum tree (get-root tree)))
	([tree node]
		(let [sentinel (get-sentinel tree)]
			(loop [max-node node]
				(let [right (get-right max-node)]
					(if (= right sentinel)
						max-node
						(recur right)))))))

;; ## successor
;; Given a node, returns the next node in an inorder walk 
;; of the tree.  Returns nil if the node it the rightmost.
(defn successor 
	([tree]
		(successor tree (get-root tree)))
	([tree node]
		(let [sentinel (get-sentinel tree) right (get-right node)]
			(if (not= right sentinel)
				(tree-minimum tree right)
				(let [x (ref node) y (ref (get-parent node))] 
					(while (and (not= @y sentinel)
					            (= @x (get-right @y))) 
						(dosync
							(ref-set x @y)
							(ref-set y (get-parent @y))))
					@y)))))

;; ## inorder-walk
;; Walk the tree from left to right, printing out the keys
;; as we go.
(defn inorder-walk 
	([tree]
		(inorder-walk tree (get-root tree)))
	([tree node]
		(if (not= node (get-sentinel tree))
			(do
				(inorder-walk tree (get-left node))
				(println (get-key node) " " (get-interval node))
				(inorder-walk tree (get-right node))))))

;; ## point-query
;; This function accepts a tree and a real number and returns all 
;; interval value pairs where each interval contains that number. 
(defn point-query 
	([tree point] 
		(let [results (ref [])]
			(point-query tree (get-root tree) point results)
			@results))
	([tree node point results]
		(if (not= node (get-sentinel tree))
			(let [interval (get-interval node)]
				(if (<= point (get-max node))
					(point-query tree (get-left node) point results))
				(if (interval-tree.interval/contains interval point)
					(add-to-result node results))
				(if (>= point (low interval))
					(point-query tree (get-right node) point results))))))

;; ## interval-query
;; This function accepts a tree and an interval and returns all
;; interval value pairs which overlap the given interval.
(defn interval-query 
	([tree interval]
		(let [results (ref [])]
			(interval-query tree (get-root tree) interval results)
			@results))
	([tree node interval results]
		(if (not= node (get-sentinel tree))
			(let [node-interval (get-interval node)]
				(if (<= (high interval) (get-max node))
					(interval-query tree (get-left node) interval results))
				(if (interval-tree.interval/intersects? node-interval interval)
					(add-to-result node results))
				(if (>= (low interval) (low node-interval))
					(interval-query tree (get-right node) interval results))))))

;; ## delete-fixup
;; Rebalance the tree after deletion (CLRS pg. 289).
(defn delete-fixup [tree node]
	(let [x (ref node) w (ref nil) root (get-root tree) sentinel (get-sentinel tree)]
		(while (and (not= @x root) (= (get-color @x) black))
			(if (= @x (get-left (get-parent @x)))
				(do
					(ref-set w (get-right (get-parent @x)))
					(if (= (get-color @w) red)
						(do
							(set-color @w black)
							(set-color (get-parent @x) red)
							(left-rotate tree (get-parent @x))
							(ref-set w (get-right (get-parent @x)))))
					(if (and (= (get-color (get-left @w)) black) 
					         (= (get-color (get-right @w)) black))
						(do
							(set-color @w red)
							(ref-set x (get-parent @x)))
						(do
							(if (= (get-color (get-right @w)) black)
								(do
									(set-color (get-left @w) black)
									(set-color @w red)
									(right-rotate tree @w)
									(ref-set w (get-right (get-parent @x)))))
							(set-color @w (get-color (get-parent @x)))
							(set-color (get-parent @x) black)
							(set-color (get-right @w) black)
							(left-rotate tree (get-parent @x))
							(ref-set x root))))
				(do
					(ref-set w (get-left (get-parent @x)))
					(if (= (get-color @w) red)
						(do
							(set-color @w black)
							(set-color (get-parent @x) red)
							(right-rotate tree (get-parent @x))
							(ref-set w (get-left (get-parent @x)))))
					(if (and (= (get-color (get-right @w)) black)
					         (= (get-color (get-left @w)) black))
						(do
							(set-color @w red)
							(ref-set x (get-parent @x)))
						(do
							(if (= (get-color (get-left @w)) black)
								(do
									(set-color (get-right @w) black)
									(set-color @w red)
									(left-rotate tree @w)
									(ref-set w (get-left (get-parent @x)))))
							(set-color @w (get-color (get-parent @x)))
							(set-color (get-parent @x) black)
							(set-color (get-left @w) black)
							(right-rotate tree (get-parent @x))
							(ref-set x root))))))
		(set-color @x black)))

;; ## delete
;; Takes a tree and an interval and removes the node associated
;; with the interval, if it exists in the tree (CLRS pg. 288).
(defn delete [tree interval]
	(dosync
		(let [z (ref (lookup-node tree interval))]
			(if (not= @z nil)
				(let [sentinel (get-sentinel tree) x (ref nil) y (ref nil) p (ref nil)]
					(if (or (= (get-left @z) sentinel)
							(= (get-right @z) sentinel))
						(ref-set y @z)
						(ref-set y (successor tree @z)))
					(ref-set p (get-parent @y))
					(if (not= (get-left @y) sentinel)
						(ref-set x (get-left @y))
						(ref-set x (get-right @y)))
					(set-parent @x (get-parent @y))
					(if (= (get-parent @y) sentinel)
						(set-root tree @x)
						(if (= @y (get-left (get-parent @y)))
							(set-left (get-parent @y) @x)
							(set-right (get-parent @y) @x)))
					(if (not= @y @z)
						(do
							(set-key @z (get-key @y))
							(set-interval @z (get-interval @y))
							(set-value @z (get-value @y))
							(update-max @z)
							(let [z-left (get-left @z) z-right (get-right @z)]
								(if (and (not= z-left nil) (not= z-left sentinel))
									(update-max z-left))
								(if (and (not= z-right nil) (not= z-right sentinel))
									(update-max z-right)))
							(let [parent (get-parent @z)]
								(if (and (not= parent nil) (not= parent sentinel))
									(update-max parent)))))
					(update-max-up-tree tree @p)
					(if (= (get-color @y) black)
						(delete-fixup tree @x))
					@y)))))

;; ## delete-all
;; Deletes all occurances of this interval from the tree. Intervals are treated numerically 
;; equal, so (delete tree [4 5]) will delete [4.0000 and 5.0] etc.
(defn delete-all [tree interval]
	(loop []
		(let [node (delete tree interval)]
			(if (not= node nil)
				(recur)))))















