;; This namespace defines real-valued interval objects and 
;; functions for working with them.  Intervals 
;; are represented as vectors of real values representing
;; the start and end point of the interval.
(ns interval-tree.interval
	(:gen-class))

;; ## new 
;; Create a new interval.  Takes two arguments, 
;; the low end of the interval and the high end of the interval
;; in that order and returns the vector [low high].  This function 
;; can be used anywhere an interval needs to be constructed, or 
;; the vector [low high] can be used instead.
(defn new [low high] [low high])
 
;; ## low
;; Get the numerical value of the low end of the interval
(defn low [self] (self 0))

;; ## high
;; Get the numerical value of the high end of the interval
(defn high [self] (self 1))

;; ## overlaps? 
;; Takes two intervals and returns true if they overlap, false otherwise
(defn overlaps? [this that]
	(if (or (<= (low that) (high this) (high that))
	        (<= (low that) (low this) (high that))) 
		true 
		false))

;; ## includes?
;; Takes two intervals and returns true if the first interval
;; contains the second, false otherwise
(defn includes? [this that]
	(if (<= (low this) (low that) (high that) (high this))
		true 
		false))

;; ## intersects?
;; Takes two intervals and determins if they have a non-trivial intersection,
;; return true if so, false otherwise.
(defn intersects? [this that]
	(if (or (overlaps? this that) 
	        (includes? this that) 
	        (includes? that this))
		true 
		false))

;; ## left?
;; Takes two intervals and returns true if the first is 
;; entirely to the left of the second, false otherwise
(defn left? [this that]
	(if (< (high this) (low that))
		true 
		false))

;; ## right?
;; Takes two intervals and returns true if the first is 
;; entirely to the right of the second, falso otherwise
(defn right? [this that]
	(if (> (low this) (high that))
		true 
		false))

;; ## to-string
;; Takes an interval and returns a string representation of the interval
;; as two comma separated values
(defn to-string [interval]
	(str (low interval) "," (high interval)))

;; ## contains
;; Takes an interval and a real number and returns true if the real number
;; is within the interval, false otherwise
(defn contains [interval point]
	(if (and (>= point (low interval))
	         (<= point (high interval)))
		true 
		false))

