(ns invetica.spec.elasticsearch
  "Perfunctory docstring."
  (:require
   [clojure.spec :as s]
   [clojure.spec.gen :as sgen]
   [invetica.spec :as spec]))

(def ^:private types
  {:hierarchal #{::nested
                 ::object}
   :simple #{::boolean
             ::date
             ::double
             ::ip
             ::keyword
             ::long
             ::text}
   :specialized #{::completion
                  ::geo_point
                  ::geo_shape}})

(s/def ::type
  (into #{} (comp cat (map name)) (vals types)))

(s/def ::enabled
  boolean?)

(s/def ::_all
  (s/keys :req-un [::enabled]))

;; -----------------------------------------------------------------------------
;; Boolean

(s/def ::boolean
  #{{:type "boolean"}})

;; -----------------------------------------------------------------------------
;; Completion

(s/def ::analyzer
  #{"english"
    "fingerprint"
    "french"
    "keyword"
    "simple"
    "stop"
    "whitespace"})

(s/def ::max-input-length
  pos-int?)

(s/def ::preserve-position-increments
  boolean?)

(s/def ::preserve-separators
  boolean?)

(s/def ::search-analyzer
  #{"analyzer"})

(s/def ::completion
  (s/with-gen (s/keys :req-un [::type]
                      :opt-un [::analyzer
                               ::max-input-length
                               ::preserve-position-increments
                               ::preserve-separators
                               ::search-analyzer])
    (fn []
      (sgen/fmap
       #(assoc % :type "completion")
       (spec/opt-hash-map #{::analyzer
                            ::max-input-length
                            ::preserve-position-increments
                            ::preserve-separators
                            ::search-analyzer})
       #_(sgen/hash-map
          :analyzer (s/gen (s/nilable ::analyzer))
          :max-input-length (s/gen (s/nilable ::max-input-length))
          :preserve-position-increments (s/gen (s/nilable ::preserve-position-increments))
          :preserve-separators (s/gen (s/nilable ::preserve-separators))
          :search-analyzer (s/gen (s/nilable ::search-analyzer)))))))

;; -----------------------------------------------------------------------------
;; Mappings

(s/def ::mapping-name :invetica.spec/some-string-alphanumeric)

(s/def ::mapping
  (s/or :boolean ::boolean
        :completion ::completion
        :date ::date
        :double ::double
        :geo-point ::geo-point
        :geo-shape ::geo-shape
        :ip ::ip
        :keyword ::keyword
        :long ::long
        :nested ::nested
        :object ::object
        :text ::text))

(s/def ::mappings
  (s/map-of ::mapping-name ::mapping))
