(ns adstage-clj.notify
  (:import (java.io StringWriter PrintWriter))
  (:require [clojure.data.json :as json]
            [environ.core :refer [env]]
            [org.httpkit.client :as http]))

(def adstage-notify-event-url
  (atom (env :adstage-notify-event-url)))
(def adstage-notify-error-url
  (atom (env :adstage-notify-error-url)))

(defn notify-event [event-data]
  "Send POST to adstage-notify-event-url with JSON body made from an arbitrary
   event-data map."
  (http/post @adstage-notify-event-url
    {:timeout 1000
     :headers {"Content-Type" "application/json"}
     :body (json/write-str event-data)}))

(defn- extract-stacktrace [exception]
  (let [string-writer (StringWriter.)
        print-writer (PrintWriter. string-writer true)]
    (.printStackTrace exception print-writer)
    (-> string-writer .getBuffer .toString)))

(defn notify-error
  "Send POST to adstage-notify-error-url with JSON body containing
   details about the error."
  ([exception] (notify-error exception nil nil))
  ([exception data] (notify-error exception data nil))
  ([exception data request]
    (http/post @adstage-notify-error-url
     {:timeout 1000
      :headers {"Content-Type" "application/json"}
      :body (json/write-str {
       "error_type" (-> exception .getClass .getName)
       "error_message" (.getMessage exception)
       "error_trace" (extract-stacktrace exception)
       "request" request
       "data" data
       "app" (env :app)
       "component" (env :com)
       "service" (env :ser)
       "environment" (env :env)
       "version" (env :ver)})})))

(defmacro install-default-error-handler!
  "Sets the default thread error handler for new threads. Will not affect main
   or other currently running threads. Is a macro so log-function can be a
   macro if you need it to be."
  ([] `(install-default-error-handler! println))
  ([log-function]
    `(Thread/setDefaultUncaughtExceptionHandler
      (reify Thread$UncaughtExceptionHandler
        (~'uncaughtException [~'_ thread# ex#]
          (~log-function ex# "Uncaught exception on" (.getName thread#))
          (notify-error ex#))))))
