(ns bloom.commons.aws.s3.core
  (:require
   [clojure.data.json :as json]
   [clojure.data.xml :as xml]
   [clojure.string :as string]
   [com.rpl.specter :as x]
   [org.httpkit.client :as http]
   [bloom.commons.specter :refer [key=]]
   [bloom.commons.aws.s3.core :as aws])
  (:import
   (java.time.temporal ChronoUnit)
   (org.apache.commons.codec.binary Base64 Hex)))

(defn generate-s3-upload-policy
  [config {:keys [group-id user-id]}]
  (let [{:keys [api-secret api-key region bucket]} config]
    (when api-secret
      (let [utc-now (aws/utc-now)
            day (.format utc-now aws/basic-date-format)
            date (.format utc-now aws/basic-date-time-format)
            credential (->> [api-key day region "s3" "aws4_request"]
                            (string/join "/"))
            policy (-> {:expiration (-> (.plus utc-now 5 ChronoUnit/MINUTES)
                                        (.format aws/date-time-format))
                        :conditions
                        [{:bucket bucket}
                         ["starts-with" "$key" (str group-id "/" user-id "/")]
                         {:acl "private"}
                         ["starts-with" "$Content-Type" ""]
                           ;; 200mb max upload size - is this resonable?
                         ["content-length-range" 0 (* 500 1024 1024)]

                         {"x-amz-algorithm" "AWS4-HMAC-SHA256"}
                         {"x-amz-credential" credential}
                         {"x-amz-date" date}]}
                       json/write-str
                       aws/str->bytes
                       Base64/encodeBase64String)]
        {:bucket bucket
         :region region
         :auth {:policy policy
                :key api-key
                :signature (->>
                            (aws/str->bytes policy)
                            (aws/hmac-sha256
                             (aws/signing-key
                              {:aws-api-secret api-secret
                               :aws-region region
                               :day day
                               :service "s3"}))
                            Hex/encodeHexString)
                :credential credential
                :date date}}))))

(defn s3-host
  [config]
  (let [{:keys [bucket region]} config]
    (str bucket ".s3" (when (not= region "us-east-1")
                        (str "." region))
         ".amazonaws.com")))

(defn get-signature
  [config utc-now path query-str]
  (let [{:keys [region api-secret bucket]} config]
    (->> ["AWS4-HMAC-SHA256"
          (.format utc-now aws/basic-date-time-format)
          (string/join "/" [(.format utc-now aws/basic-date-format) region "s3"
                            "aws4_request"])
          (aws/canonical-request
           {:method "GET"
            :path (str "/" path)
            :query-string query-str
            :headers {"host" (str bucket ".s3.amazonaws.com")}
            :body nil})]
         (string/join "\n")
         aws/str->bytes
         (aws/hmac-sha256
          (aws/signing-key {:aws-api-secret api-secret
                            :aws-region region
                            :day (.format utc-now aws/basic-date-format)
                            :service "s3"}))
         aws/bytes->hex)))

(defn readable-s3-url
  [config path]
  (let [{:keys [api-key region]} config
        utc-now (aws/utc-now)
        query-str (str "X-Amz-Algorithm=AWS4-HMAC-SHA256"
                       "&X-Amz-Credential=" api-key "/" (.format utc-now aws/basic-date-format) "/" region "/s3/aws4_request"
                       "&X-Amz-Date=" (.format utc-now aws/basic-date-time-format)
                       "&X-Amz-Expires=604800"  ; seven days
                       "&X-Amz-SignedHeaders=host")]
    (str "https://" (s3-host config)
         "/" path
         "?" query-str
         "&X-Amz-Signature=" (get-signature config utc-now path query-str))))

(defn- make-request
  [config {:keys [body method path] :as request}]
  (let [utc-now (aws/utc-now)
        {:keys [region api-secret api-key bucket]} config
        req (update request :headers
                    assoc
                    "x-amz-date" (.format utc-now aws/basic-date-time-format)
                    "x-amz-content-sha256" (aws/hex-hash body)
                    "Host" (str (s3-host config) ":443"))]
    (-> req
        (dissoc :method :path :query-string)
        (assoc-in [:headers "Authorization"]
                  (aws/auth-header {:now utc-now
                                    :service "s3"
                                    :request req
                                    :aws-api-secret api-secret
                                    :aws-api-key api-key
                                    :aws-region region})))))

(defn delete-file!
  [config path]
  (let [{:keys [api-secret api-key region bucket]} config]
    @(http/delete (str "https://" (s3-host config) "/" path)
                  (make-request config {:method "DELETE"
                                        :path (str "/" path)
                                        :body ""}))))

(defn find-thumbnails
  [config video-path]
  (let [{:keys [api-secret api-key region bucket]} config
        prefix (string/replace video-path #"\.mp4" "-")
        query-str (str "list-type=2&prefix=" prefix)]
    (some->>
     @(http/get (str "https://" (s3-host config) "/?" query-str)
                (make-request config {:method "GET"
                                      :query-string query-str
                                      :body ""
                                      :path "/"}))
     :body
     xml/parse-str
     (x/select [:content x/ALL
                (key= :tag :Contents) :content
                x/FIRST (key= :tag :Key)
                :content x/FIRST
                #(string/ends-with? % ".jpg")])
     sort)))
