(ns bloom.commons.config
  (:refer-clojure :exclude [read])
  (:require
   [clojure.java.io :as io]
   [clojure.pprint :as pprint]
   [clojure.tools.reader.edn :as edn]
   [malli.core :as malli]
   [malli.util :as malli.util]
   [malli.error :as malli.error]
   [malli.generator :as malli.generator]
   [zprint.core :as zprint]))

(defn ^:private parse-file
  [path]
  (when (.exists (io/file path))
    (->> path
         slurp
         edn/read-string)))

(defn parse [config schema]
  (if (malli/validate (malli.util/closed-schema schema) config)
    config
    (let [cause (malli/explain (malli.util/closed-schema schema) config)]
      (throw
       (ex-info (str "Config invalid\n"
                     (with-out-str
                       (pprint/pprint (malli.error/humanize cause)))) cause)))))

(defn read [source-path schema]
  (if-let [config (parse-file source-path)]
    (parse config schema)
    (throw
     (ex-info (str "Config file missing (" source-path ")") {}))))

(defn generate [source-path schema]
  (if (.exists (io/file source-path))
    (throw
      (ex-info (str "Config file already exists. Not generating new one.") {}))
    (spit source-path
          (with-out-str
            (zprint/zprint
              (malli.generator/generate schema {:size 0})
              {:style :community
               :width 160
               :set {:sort? true}
               :map {:comma? false
                     :sort? true
                     :lift-ns? false
                     :force-nl? true}})))))
