(ns bloom.commons.pages
  ";; call at some point to initialize:

   (initialize! [{:page/id :home
                  :page/view #'some-view-fn
                  :page/path \"/\"}
                 {:page/id :profile
                  :page/view (fn [[page-id {:keys [id]}]] ...)
                  :page/path \"/profile/:id\"
                  :page/parameters {:id integer?}}
                 ...])

   ;; include a reagent-view somewhere:
   [current-page-view]

   ;; to generate link string:
   (path-for [:profile {:id 5}])

   ;; to force navigation to some page
   (navigate-to! [:profile {:id 5}])

   ;; to check if some page is active
   (active? [:profile])
   (active? [:profile {:id 5}])
  "
  (:require
    [clojure.set :as set]
    [malli.core :as m]
    [reitit.core :as reitit]
    [reitit.coercion :as coercion]
    [reitit.coercion.malli]
    [reitit.impl]
    [bloom.commons.uri :as uri]
    #?@(:cljs
         [[reagent.core :as r]
          [accountant.core :as accountant]])))

(defn classify-parameters
  "/a/:bar   {:bar __, :foo ___}
  =>
  {:path {:bar ___}, :query {:foo ___}}

  (assumes any params not in path are query params)"
  [path parameters]
  (let [children (if parameters (m/children parameters) [])
        all-param-keys (set (map first children))
        path-param-keys (:path-params (reitit.impl/parse path {}))
        query-param-keys (set/difference all-param-keys path-param-keys)]
    {:path (into [:map] (->> children
                             (filter (fn [[k _]]
                                       (contains? path-param-keys k)))))
     :query (into [:map] (->> children
                              (filter (fn [[k _]]
                                       (contains? query-param-keys k)))))}))

(defn ->args [page]
  [(get-in page [:data :config :page/id])
   (merge (get-in page [:coerced-parameters :query])
          (get-in page [:coerced-parameters :path]))])

(defn make-router [pages]
  (reitit/router
    (->> pages
         (map (fn [page]
                [(page :page/path)
                 {:name (page :page/id)
                  :coercion reitit.coercion.malli/coercion
                  :parameters (classify-parameters (page :page/path) (page :page/parameters))
                  :config page}])))
    {:compile coercion/compile-request-coercers}))

(defn path-for*
  [router [page-id params]]
  (let [match (reitit/match-by-name router page-id params)
        query-params (->> (get-in match [:data :parameters :query])
                          (m/children)
                          (map first)
                          (select-keys params))]
    (uri/to-string {:path (:path match)
                    :query-params query-params})))

(defn active?*
  [match [page-id parameters]]
  (= (->args match)
     [page-id parameters])
  #_(and
    (= page-id
       (get-in match [:data :config :page/id]))
    (if parameters
      (= parameters
         (select-keys (merge (get-in match [:coerced-parameters :query])
                             (get-in match [:coerced-parameters :path]))
                      (keys parameters)))
      true)))

(defn match-by-path
  "Unlike reitit/match-by-path, path allows for query-params"
  [router path]
  (let [uri (uri/from-string path)
        ;; path does not include query-params
        path-name (:path uri)]
    (when-let [match (reitit/match-by-path router path-name)]
      (let [match (assoc match :query-params (:query-params uri))]
        (assoc match :coerced-parameters (coercion/coerce! match))))))

(defn page-by-path
  [router path]
  (get-in (match-by-path router path) [:data :config]))

#?(:cljs
   (do
     (defonce initialized? (atom false))
     (defonce router (atom nil))
     (defonce current-match (r/atom nil))

     (defn initialize!
       "Expects a list of pages, each a map with the following keys:
       :page/id           keyword, used in path-for

       :page/view         reagent view fn (recommend using #'view-fn)
       receives params of page

       :page/path         string defining path
       may include param patterns in path ex. /foo/:id
       (params must also be included in :page/parameters)

       :page/parameters   map, malli coercion of path
       any parameters not included in :page/path are assumed to be query-params

       :page/on-enter!    fn to call when page is navigated to
       (right after reagent state is updated with new page-id)
       receives params (of new page): [page-id params]

       :page/on-exit!     fn to call when page is navigated away from
       receives params (of old page): [page-id params]"
       [pages]
       (when-not @initialized?
         (reset! initialized? true)

         (reset! router (make-router pages))

         (accountant/configure-navigation!
           {:nav-handler (fn [path]
                           (when-let [on-exit! (get-in @current-match [:data :config :page/on-exit!])]
                             (on-exit! (->args @current-match)))

                           (when-let [match (match-by-path @router path)]
                             (reset! current-match match)
                             (when-let [on-enter! (get-in @current-match [:data :config :page/on-enter!])]
                               (on-enter! (->args @current-match)))))
            :path-exists? (fn [path]
                            (boolean (reitit/match-by-path @router (:path (uri/from-string path)))))})
         (accountant/dispatch-current!)))

     (defn current-page-view []
       (when-let [view (get-in @current-match [:data :config :page/view])]
         [view (->args @current-match)]))

     (defn path-for
       [[page-id params]]
       (path-for* @router [page-id params]))

     (defn navigate-to!
       [[page-id params]]
       (accountant/navigate! (path-for [page-id params])))

     (defn active?
       [[page-id parameters]]
       (active?* @current-match [page-id parameters]))))

