(ns bloom.commons.magic-token
  (:require
    [bloom.commons.crypto :as crypto])
  (:import
    (java.time Instant)
    (java.security SecureRandom)))

(defn ^:private rand-ints
  [n]
  (repeatedly n (fn [] (.nextInt (SecureRandom.) 9))))

(defn ^:private remove-expired [tokens-kv now]
  (->> tokens-kv
       (remove (fn [[_ {::keys [expires-at]}]]
                 (.isBefore expires-at now)))
       (into {})))

(defn ^:private remove-expired! [store]
  (swap! store update ::tokens remove-expired (Instant/now)))

(defn ^:private mark-success!
  [store user-id]
  (if (pos? (get-in @store [::tokens user-id ::successes-left]))
    (swap! store update-in [::tokens user-id ::successes-left] dec)
    (swap! store update ::tokens dissoc user-id)))

(defn ^:private mark-failure!
  [store user-id]
  (if (pos? (get-in @store [::tokens user-id ::failures-left]))
    (swap! store update-in [::tokens user-id ::failures-left] dec)
    (swap! store update ::tokens dissoc user-id)))

(defn init!
  "Generates a magic-token store, with configurable ttl-ms and token-length."
  [{:keys [ttl-ms token-length failure-max success-max] :as opts}]
  {:pre [(integer? ttl-ms)
         (integer? token-length)
         (integer? failure-max)
         (integer? success-max)]}
  (atom {::opts opts
         ::tokens {}}))

(defn generate!
  "For a user-id, returns a short-lived token (a list with N single-digit integers);
    that can be later verified with valid?().
   If called multiple times with the same user-id, the token is reset (with new ttl, fresh attempt pool, etc.)."
  [store user-id]
  (let [opts (@store ::opts)
        token (rand-ints (opts :token-length))]
    (remove-expired! store)
    (swap! store assoc-in [::tokens user-id]
           {::token token
            ::failures-left (dec (opts :failure-max))
            ::successes-left (dec (opts :success-max))
            ::expires-at (.plusMillis (Instant/now)
                                      (get-in @store [::opts :ttl-ms]))})
    token))

(defn valid?
  "For a user-id, verifies a token (list with N single-digit integers) is valid
    (matches generated token, is not yet expired, and has not reached max fails).
    Can be called multiple times, but to a limit of configured max-fails and max-success."
  [store user-id token]
  (remove-expired! store)
  (if-let [previous-token (get-in @store [::tokens user-id ::token])]
    (let [matches? (crypto/slow= token previous-token)]
      (if matches?
        (mark-success! store user-id)
        (mark-failure! store user-id))
      matches?)
    false))
