(ns bloom.commons.cache
  "Provides a caching middleware that caches GET requests to files.
    (make-cache-middleware {:path \"./some-folder\"})
  Cache can be skipped by doing a hard refresh.
  Basically, a simplified in-process version of scrip."
  (:require
    [byte-streams :as bs]
    [bloom.commons.thread-safe-io :as tsio]
    [clojure.edn :as edn]
    [clojure.java.io :as io])
  (:import
    (java.security MessageDigest)))

(defn sha1
  ^String
  [^String s]
  (-> (doto (MessageDigest/getInstance "SHA-1")
        (.update (.getBytes s "UTF-8" )))
      (.digest)
      ;; to hex string
      (->> (map (partial format "%02x"))
           (apply str))))

(defn make-folders!
  [{:keys [path]}]
  (.mkdir (io/file path))
  (.mkdir (io/file path "meta"))
  (.mkdir (io/file path "body")))

(defn ->cache-key
  [request]
  (sha1 (str (request :request-method)
             (request :source)
             (request :uri)
             (request :query-string))))

(defn ->meta-file
  [{:keys [path]} request]
  (io/file path "meta" (->cache-key request)))

(defn ->body-file
  [{:keys [path]} request]
  (io/file path "body" (->cache-key request)))

(defn cacheable?
  [request]
  (= :get (:request-method request)))

(defn skip-cache?
  [request]
  (or (= "no-cache" (get-in request [:headers "pragma"]))
      (= "no-cache" (get-in request [:headers "cache-control"]))))

(defn from-cache
  [config request]
  (-> (edn/read-string (tsio/slurp (->meta-file config request)))
      (assoc-in [:headers "X-Cache-Status"] "from-cache")
      (assoc :body (io/input-stream (->body-file config request)))))

(defn cached?
  [config request]
  (.exists (->meta-file config request)))

(defn cache!
  [config request response]
  (when (and
          (:status response)
          (<= 200 (:status response) 299))
    (with-open [wrtr (io/output-stream (->body-file config request))]
      (.write wrtr (bs/to-byte-array (:body response))))
    (tsio/spit (->meta-file config request)
               (dissoc response :body))))

(defn purge!
  [{:keys [path]}]
  ;; this is not thread safe
  (doseq [f (file-seq (io/file path "meta"))]
    (.delete f))
  (doseq [f (file-seq (io/file path "body"))]
    (.delete f)))

(defn make-cache-middleware
  [{:keys [path] :as config}]
  (make-folders! config)
  (fn [handler]
    (fn [request]
      (if (and
            (cacheable? request)
            (not (skip-cache? request))
            (cached? config request))
        (from-cache config request)

        (let [response (handler request)]
          (when (cacheable? request)
            (future (cache! config request response)))
          response)))))
