(ns bloom.commons.uri
  (:require
    [clojure.string :as string]
    #?@(:clj
         [[ring.util.codec :as codec]]))
  (:import
    #?@(:cljs
         [(goog Uri)]
         :clj
         [(java.net URI)])))

(defn coerce-to-string [o]
  (cond
    (keyword? o)
    (name o)

    :else
    (str o)))

(defn url-encode [value]
  #?(:clj (codec/url-encode (coerce-to-string value))
     :cljs (js/encodeURIComponent (string/replace (coerce-to-string value) "%20" "+"))))

(defn params->query-string
  "Given map of params (allowing array values), returns a query string;
   Keys are sorted, and values within a key are sorted.
   (sorting is done to improve reproducability/cacheability)

   Ex.
     {:a 1
      :b [2 3]}
    =>
   a=1&b[]=2&b[]=3"
  [params]
  (->> params
       sort
       (mapcat (fn [[k v]]
                 (cond
                   (coll? v)
                   (for [v' (sort v)]
                     (str (name k) "[]" "=" (url-encode v')))
                   :else
                   [(str (name k) "=" (url-encode v))])))
       (string/join "&")))

(defn query-string->params
  "Parses a query-string (allowing array values), returns a map.

   Ex.
     a=1&b[]=2&b[]=3
    =>
     {:a 1
      :b [2 3]}"
  [query-string]
  (->> (string/split query-string #"&")
       (map (fn [kv]
              (string/split kv #"=")))
       (group-by first)
       (map (fn [[k kvs]]
              (if (string/ends-with? k "[]")
                [(keyword (string/replace k #"\[\]$" ""))
                 (mapv second kvs)]
                [(keyword k)
                 (second (first kvs))])))
       (into {})))

(defn from-string [uri]
  #?(:clj
     (let [java-uri (URI. uri)]
       {:path (.getPath java-uri)
        :query-params (query-string->params (.getQuery java-uri))})
     :cljs
     (let [goog-uri (.parse Uri uri)]
       {:path (.getPath goog-uri)
        :query-params (query-string->params (.getQuery goog-uri))})))

(defn to-string [uri]
  (str (:path uri)
       (when (seq (:query-params uri))
         (str "?" (params->query-string (:query-params uri))))))
