(ns bloom.commons.thread-safe-io
  (:refer-clojure :exclude [spit slurp])
  (:require
   [clojure.core.memoize :as memoize]
   [clojure.java.io :as io]))

(defn -file-agent
  [file-name]
  (add-watch
   (agent nil) :file-writer
   (fn [_ _ _ [op cb ?arg]]
     (cb
      (try
        (case op
          :write (with-open [w (io/writer file-name)]
                   (.write w (str ?arg))
                   (.flush w))
          :read (clojure.core/slurp file-name))
        (catch java.io.FileNotFoundException e
          e))))))

;; after 60s of inactivity, remove the agent
;; important: if a file takes longer than 60s to read or write,
;; the next read or write operation is not thread-safe
(def file-agent (memoize/ttl -file-agent :ttl/threshold (* 60 1000)))

(defn spit
  [file-name content]
  (let [p (promise)]
    (send (file-agent file-name)
          (constantly [:write (fn [result]
                                (deliver p result))
                       content]))
    (let [result @p]
      (if (instance? Throwable result)
        (throw result)
        result))))

(defn slurp
  [file-name]
  (let [p (promise)]
    (send (file-agent file-name)
          (constantly [:read (fn [content]
                               (deliver p content))]))
    (let [result @p]
      (if (instance? Throwable result)
        (throw result)
        result))))

#_(do
    (clojure.core/spit "fstest/test.edn" (repeat 100000 -1))
    (dotimes [n 100]
      (future (spit "fstest/test.edn" (repeat 100000 n)))
      (future (clojure.core/spit (str "fstest/out" n)
                                 (try
                                   ((juxt (partial apply =)
                                          first)
                                    (read-string (slurp "fstest/test.edn")))
                                   (catch Throwable e
                                     e))))))
