(ns bloom.commons.thread-safe-io
  (:refer-clojure :exclude [spit slurp])
  (:require
   [clojure.core.memoize :as memoize]
   [clojure.java.io :as io]))

(defn -file-agent
  [file-name]
  (add-watch
   (agent nil) :file-writer
   (fn [_ _ _ [op cb ?arg]]
     (cb
      (try
        (case op
          :write (with-open [w (io/writer file-name)]
                   (.write w (str ?arg))
                   (.flush w))
          :read (clojure.core/slurp file-name)
          :update (let [prev (clojure.core/slurp file-name)]
                    (with-open [w (io/writer file-name)]
                      (.write w (str (?arg prev)))
                      (.flush w))))
        (catch java.io.FileNotFoundException e
          e))))))

;; after 60s of inactivity, remove the agent
;; important: if a file takes longer than 60s to read or write,
;; the next read or write operation is not thread-safe
(def file-agent (memoize/ttl -file-agent :ttl/threshold (* 60 1000)))

(defn- blocking-send
  [file-name op & [?arg]]
  (let [p (promise)]
    (send (file-agent file-name)
          (constantly [op (fn [result] (deliver p result))
                       ?arg]))
    (let [result @p]
      (if (instance? Throwable result)
        (throw result)
        result))))

(defn spit
  [file-name content]
  (blocking-send file-name :write content))

(defn slurp
  [file-name]
  (blocking-send file-name :read))

(defn transact
  [file-name update-fn]
  (blocking-send file-name :update update-fn))

#_(do
    (.mkdirs (io/file "fstest"))
    (clojure.core/spit "fstest/test.edn" (repeat 100000 -1))
    (dotimes [n 100]
      (future (spit "fstest/test.edn" (repeat 100000 n)))
      (future (clojure.core/spit (str "fstest/out" n)
                                 (try
                                   ((juxt (partial apply =)
                                          first)
                                    (read-string (slurp "fstest/test.edn")))
                                   (catch Throwable e
                                     e))))))
