(ns bloom.commons.memoize
  (:require
   [duratom.core :as duratom]
   [clojure.java.io :as io]
   [clj-commons.digest :as digest]
   [taoensso.nippy :as nippy]))

;; postgres
;; nippy files
;; duratom

(defn memo
  [{:keys [calculate cache-key lookup store!]}]
  (fn [& args]
    (let [k (cache-key args)]
      (if-let [result (lookup k)]
        result
        (let [result (apply calculate args)]
          (store! k result)
          result)))))

(defn duratom-memo
  [{:keys [calculate file-path]}]
  (let [da (duratom/duratom :local-file
                            :file-path file-path
                            :init {})]
    (memo {:calculate calculate
           :cache-key (fn [args]
                        args)
           :lookup (fn [k]
                     (@da k))
           :store! (fn [k value]
                     (swap! da assoc k value))})))

(extend-protocol
  digest/Digestible
  clojure.lang.PersistentArrayMap
  (-digest [message algorithm]
    (digest/-digest (pr-str message) algorithm)))

(defn nippy-memo
  [{:keys [calculate file-path]}]
  (let [->f (fn [k]
              (io/file file-path (str k ".nippy")))]
    (memo {:calculate calculate
           :cache-key (fn [args]
                        (digest/sha-256 args))
           :lookup (fn [k]
                     (let [f (->f k)]
                       (when (.exists f)
                         (nippy/thaw-from-file f))))
           :store! (fn [k value]
                     (let [f (->f k)]
                       (.mkdirs (.getParentFile f))
                       (nippy/freeze-to-file f value)))})))

