(ns bloom.commons.pages
  ";; call at some point to initialize:

   (initialize! [{:page/id :home
                  :page/view #'some-view-fn
                  :page/path \"/\"}
                 {:page/id :profile
                  :page/view (fn [[page-id {:keys [id]}]] ...)
                  :page/path \"/profile/:id\"
                  :page/parameters {:id integer?}}
                 ...])

   ;; can also register pages independently

   (register-page! {:page/id :other ...})

   ;; include a reagent-view somewhere:
   [current-page-view]

   ;; to generate link string:
   (path-for [:profile {:id 5}])

   ;; to force navigation to some page
   (navigate-to! [:profile {:id 5}])

   ;; to check if some page is active
   (active? [:profile])
   (active? [:profile {:id 5}])
  "
  (:require
    [clojure.set :as set]
    [clojure.string :as string]
    [reitit.core :as reitit]
    [reitit.coercion :as coercion]
    [reitit.coercion.malli]
    [reitit.impl]
    [malli.core :as m]
    [malli.experimental.lite :as malli.lite]
    [bloom.commons.query-params :as query-params]
    #?@(:cljs
         [[accountant.core :as accountant]
          [reagent.core :as r]]))
    #?(:cljs
       (:import
         (goog Uri))))

(defonce initialized? (atom false))
(defonce router (atom nil))
(defonce pages
  #?(:cljs (r/atom {})
     :clj (atom {})))
(defonce current-page
  #?(:cljs (r/atom nil)
     :clj (atom nil)))

(defn classify-parameters
  "/a/:bar   {:bar __, :foo ___}
  =>
  {:path {:bar ___}, :query {:foo ___}}

  (assumes any params not in path are query params)"
  [path parameters]
  (let [children (if parameters (m/children parameters) [])
        all-param-keys (set (map first children))
        path-param-keys (:path-params (reitit.impl/parse path {}))
        query-param-keys (set/difference all-param-keys path-param-keys)]
    {:path (into [:map] (->> children
                             (filter (fn [[k _]]
                                       (contains? path-param-keys k)))))
     :query (into [:map] (->> children
                              (filter (fn [[k _]]
                                       (contains? query-param-keys k)))))}))

(defn ->args [current-page]
  [(get-in current-page [:data :config :page/id])
   (merge (get-in current-page [:coerced-parameters :query])
          (get-in current-page [:coerced-parameters :path]))])

(defn request->page
  [{:keys [uri query-params]}]
  (when-let [match (reitit/match-by-path @router uri)]
    (let [match (assoc match :query-params (or query-params {}))]
      (assoc match :coerced-parameters (coercion/coerce! match)))))

(defn- maybe-update-current-page!
  "Updates current-page if path+query-params match a page; returns the page if matches, otherwise nil."
  [{:keys [uri query-params] :as request}]
  (when-let [page (request->page request)]
    (reset! current-page page)))

#?(:cljs
   (do
     (defn path-exists? [path]
       (let [uri (.parse Uri path)]
         (boolean (reitit/match-by-path @router (.getPath uri)))))

     (defn nav-handler [path]
       (when-let [on-exit! (get-in @current-page [:data :config :page/on-exit!])]
         (on-exit! (->args @current-page)))

       (let [uri (.parse Uri path)]
         (when (maybe-update-current-page!
                 {:uri (.getPath uri)
                  :query-params (query-params/url->params uri)})
           (when-let [on-enter! (get-in @current-page [:data :config :page/on-enter!])]
             (on-enter! (->args @current-page))))))

     (defn configure-accountant! []
       (accountant/configure-navigation!
         {:nav-handler nav-handler
          :path-exists? path-exists?}))))

(defn initialize!
  "Expects a list of pages. See register-page! for supported page parameters.
   Does not need be called if used only on server-side, can just use register-page!
   Should be called once in cljs."
  [input-pages]
  (when-not @initialized?
    (reset! initialized? true)

    (swap! pages merge (zipmap (map :page/id input-pages)
                               input-pages))

    #?(:cljs
       (do
         (configure-accountant!)
         (accountant/dispatch-current!)))))

#?(:clj
   (defn set-current-page! [request]
     (maybe-update-current-page! request)))

(defn register-page!
  "Add a page, supports parameters:

     :page/id           keyword, used in path-for

     :page/view         reagent view fn (recommend using #'view-fn)
                        receives params of page

     :page/path         string defining path
                        may include param patterns in path ex. /foo/:id
                        (params must also be included in :page/parameters)

     :page/override-conflicting-path?
                        boolean, override reitit exception for conflicting paths

     :page/parameters   vector or map (lite) syntax for malli
                        any parameters not included in :page/path are assumed to be query-params

     :page/on-enter!    fn to call when page is navigated to
                        (right after reagent state is updated with new page-id)
                        receives params (of new page): [page-id params]

     :page/on-exit!     fn to call when page is navigated away from
                        receives params (of old page): [page-id params]"

  [page]
  (swap! pages assoc (:page/id page) page))

(defonce watcher
  (add-watch
    pages
    ::update-router
    (fn [_ _ _ latest-pages]
      (reset! router
              (reitit/router
                (->> latest-pages
                     vals
                     (sort-by :page/conflict-priority)
                     (map (fn [page]
                            [(page :page/path)
                             {:name (page :page/id)
                              :conflicting (boolean (page :page/conflict-priority))
                              :coercion reitit.coercion.malli/coercion
                              :parameters (classify-parameters (page :page/path)
                                                               (cond
                                                                 (map? (page :page/parameters))
                                                                 (malli.lite/schema (page :page/parameters))
                                                                 (vector? (page :page/parameters))
                                                                 (page :page/parameters)))
                              :config page}])))
                {:compile coercion/compile-request-coercers})))))

(defn current-page-view []
  (when-let [view (get-in @current-page [:data :config :page/view])]
    [view (->args @current-page)]))

#?(:clj
   (defn request->page-view
     [request]
     (let [page (request->page request)]
       (when-let [view (get-in page
                         [:data :config :page/view])]
         [view (->args page)]))))

#?(:cljs
   (defn dispatch-current!
     "Force re-trigerring of current page, useful for authenticating while on same page"
     []
     (accountant/dispatch-current!)))

(defn path-for
  [[page-id params]]
  (when-let [match (reitit/match-by-name @router page-id params)]
    (let [query-params (some->> (get-in match [:data :parameters :query])
                                (m/children)
                                (map first)
                                (select-keys params))]
      (str (:path match)
           (when (seq query-params)
             (let [query (query-params/params->string query-params)]
               (when-not (string/blank? query)
                 (str "?" query))))))))

#?(:cljs
   (defn navigate-to!
     [[page-id params]]
     (accountant/navigate! (path-for [page-id params]))))

(defn active?
  [[page-id parameters]]
  (and
    (= page-id
       (get-in @current-page [:data :config :page/id]))
    (if parameters
      (= parameters
         (select-keys (merge (get-in @current-page [:coerced-parameters :query])
                             (get-in @current-page [:coerced-parameters :path]))
                      (keys parameters)))
      true)))
