(ns bloom.commons.debounce)

(defn- schedule!
  [f wait]
  #?(:clj (future
            (Thread/sleep wait)
            (f))
     :cljs (js/setTimeout f wait)))

(defn- cancel!
  [t]
  #?(:clj (try
            (future-cancel t)
            (catch java.lang.InterruptedException _))
     :cljs (js/clearTimeout t)))

(defn debounce
  "Given a function and a ms timeout, returns a new fn that only calls the given fn after ms have passed since the last invocation of the new fn"
  [f wait]
  (let [timeout (atom nil)]
    (fn [& args]
      (when-let [t @timeout]
        (cancel! t))
      (reset! timeout
              (schedule! (fn [] (apply f args))
                         wait))
      nil)))

(defn debounce-by
  "Like debounce, but also takes key-f, which is used to group calls to the same function, creating seperate debounce schedules for each result of key-f"
  [f key-f wait]
  (let [timeouts (atom {})]
    (fn [& args]
      (let [k (apply key-f args)]
        (when-let [t (get @timeouts k)]
          (cancel! t))
        (swap! timeouts
               assoc
               k
               (schedule! (fn [] (apply f args))
                          wait))
        nil))))

