(ns bloom.commons.file-db
  "DB as EDN files

   db-config is a map:
    {:data-path \"path/to/folder\"}

   Entities are stored as EDN, one entity per file, stored in folders by type.
   Entities must have an id, either {:entity/id the-id} or {:____/id the-id}
   Entities must have a type, either {:entity/type :the-type} or :the-type/id"
  (:refer-clojure :exclude [get find])
  (:require
    [clojure.string :as string]
    [clojure.java.io :as io]
    [clojure.edn :as edn]
    [clojure.set :as set]
    [time-literals.read-write :as tl]
    [zprint.core :as zprint]
    [bloom.commons.thread-safe-io :as thread-safe]))

(defn ->id [entity]
  (or (entity :entity/id)
      (->> entity
           keys
           (filter (fn [k]
                     (= "id" (name k))))
           first
           (entity))))

(defn ->type [entity]
  (or (entity :entity/type)
      (->> entity
           keys
           (filter (fn [k]
                     (= "id" (name k))))
           first
           namespace
           keyword)))

(defn ->path [db-config entity]
  (str (db-config :data-path) "/" (name (->type entity)) "/" (->id entity) ".edn"))

(defn make-directories-for-file! [output-to]
  (.mkdirs (java.io.File. (.getParent (java.io.File. output-to)))))

(defn ->edn
  [data]
  (zprint/zprint-str
    data {:style :community
          :fn-map {"defn" :fn}
          :width 160
          :set {:sort? true}
          :map {:comma? false
                :sort? true
                :lift-ns? false
                :force-nl? true}}))

(defn parse-edn
  [s]
  (edn/read-string {:readers tl/tags} s))

(defn write-file!
  [path data]
  (try
    (make-directories-for-file! path)
    (thread-safe/spit path (->edn data))
    (catch java.lang.NullPointerException e
      (println "Error writing" data))))

(defn write-entity!
  [db-config entity]
  (write-file! (->path db-config entity) entity))

(defn write-entities!
  [db-config entities]
  (doseq [entity entities]
    (write-entity! db-config entity)))

(defn update-entity!
  [db-config entity f]
  (thread-safe/transact
    (->path db-config entity)
    (fn [c]
      (->> c
           parse-edn
           f
           ->edn))))

(defn ^:dynamic read-file
  [f]
  (parse-edn (thread-safe/slurp f)))

(defn ^:dynamic entity-files [db-config path]
  (->> (io/file (db-config :data-path) path)
       file-seq
       (filter (fn [f]
                 (and
                   (.isFile f)
                   (string/ends-with? (.getName f) "edn"))))))

(defn- read-entities* [db-config path]
  (->> (entity-files db-config path)
       (map read-file)
       (mapcat (fn [edn]
                  (if (vector? edn)
                    edn
                    [edn])))))

(defn all
  "Returns all entities"
  [db-config]
  (read-entities* db-config ""))

(defn all-by-type
  "Returns entities that match the given type"
  [db-config type]
  (read-entities* db-config (name type)))

(defn get
  "Returns the first entity with the given id, or nil."
  [db-config id]
  (some->> (entity-files db-config "")
           (filter (fn [f]
                     (re-matches (re-pattern (str "^" id ".edn$")) (.getName f))))
           first
           read-file))

(defn search
  "Returns entities where all opts key-values match the entity.

  If an opts value and an entity value are both sets, it matches if the opts value intersects the entity value."
  [db-config opts]
  (->> (if (opts :entity/type)
         (all-by-type db-config (opts :entity/type))
         (all db-config))
       (filter (fn [e]
                 (->> opts
                      (map (fn [[k v]]
                             (cond
                               (and (set? v) (set? (e k)))
                               (boolean (seq (set/intersection v (e k))))
                               :else
                               (= v (e k)))))
                      (every? true?))))))

(defn find
  "Like search, but returns the first matching entity."
  [db-config opts]
  (first (search db-config opts)))

(defmacro with-cache [& body]
  `(binding [read-file (memoize read-file)
             entity-files (memoize entity-files)]
     ~@body))

(comment

  ;; BAD
  (let [config {:data-path "."}
        e {:entity/id "foo"
           :x 1
           :y 1}]
    (write-entity! config e)
    ;; a
    (future
      (let [e (get config "foo")]
        (Thread/sleep 500)
        (write-entity! config (update e :x inc))))
    ;; b
    (future
      (Thread/sleep 400)
      (let [e (get config "foo")]
        (write-entity! config (update e :y inc))))

    (Thread/sleep 1000)
    (let [after (get config "foo")]
      (assert (= (:x after) 2))
      (assert (= (:y after) 2) "RACEY!")))

  ;; GOOD
  (let [config {:data-path "."}
        e {:entity/id "foo"
           :x 1
           :y 1}]
    (update-entity! config e (constantly e))
    ;; a
    (future
      (update-entity! config {:entity/id "foo"} #(update % :x inc)))
    ;; b
    (future
      (update-entity! config {:entity/id "foo"} #(update % :y inc)))

    (Thread/sleep 100)
    (let [after (get config "foo")]
      (assert (= (:x after) 2))
      (assert (= (:y after) 2))))


  (->edn (->> (range 1000)
              (map (fn [x] (prn "X = " x) x))
              (filter (fn [x] (= x 999))))))
  )
