(ns bloom.commons.html
  (:require
    [clojure.string :as string]
    [huff2.core :as h]))

(def exceptions
  ;; from https://github.com/preactjs/preact-compat/issues/222
  #{"accept-charset"
    "http-equiv"
    ;; svg
    "accent-height"
    "alignment-baseline"
    "arabic-form"
    "baseline-shift"
    "cap-height"
    "clip-path"
    "clip-rule"
    "color-interpolation"
    "color-interpolation-filters"
    "color-profile"
    "color-rendering"
    "fill-opacity"
    "fill-rule"
    "flood-color"
    "flood-opacity"
    "font-family"
    "font-size"
    "font-size-adjust"
    "font-stretch"
    "font-style"
    "font-variant"
    "font-weight"
    "glyph-name"
    "glyph-orientation-horizontal"
    "glyph-orientation-vertical"
    "horiz-adv-x"
    "horiz-origin-x"
    "marker-end"
    "marker-mid"
    "marker-start"
    "overline-position"
    "overline-thickness"
    "panose-1"
    "paint-order"
    "stop-color"
    "stop-opacity"
    "strikethrough-position"
    "strikethrough-thickness"
    "stroke-dasharray"
    "stroke-dashoffset"
    "stroke-linecap"
    "stroke-linejoin"
    "stroke-miterlimit"
    "stroke-opacity"
    "stroke-width"
    "text-anchor"
    "text-decoration"
    "text-rendering"
    "underline-position"
    "underline-thickness"
    "unicode-bidi"
    "unicode-range"
    "units-per-em"
    "v-alphabetic"
    "v-hanging"
    "v-ideographic"
    "v-mathematical"
    "vert-adv-y"
    "vert-origin-x"
    "vert-origin-y"
    "word-spacing"
    "writing-mode"
    "x-height"})

(defn kebab->camel [s]
  (if
    (or (contains? exceptions s)
        (string/starts-with? s "data-")
        (string/starts-with? s "aria-"))
    s
    (string/replace s #"-(\w+)"
                    (fn [[_ match]]
                      (string/capitalize match)))))

#_(kebab->camel "stop-color") ;; stop-color
#_(kebab->camel "data-foo") ;; data-foo
#_(kebab->camel "data-foo-bar") ;; data-foo-bar
#_(kebab->camel "aria-foo") ;; aria-foo
#_(kebab->camel "everything-else") ;; everythingElse

(defn parse-opts
  "Given map of opts, rewrites any kebab-case opts to camelCase"
  [opts]
  (->> opts
       (map (fn [[k v]]
              [(keyword (kebab->camel (name k))) v]))
       (into {})))

(defn merge-attrs
  "Merge attr values, which can be: nil, false, a string, or a collection"
  [a b]
  (cond
    (or (false? a) (nil? a))
    b
    (or (false? b) (nil? b))
    a
    :else
    (let [a (if (string? a) [a] a)
          b (if (string? b) [b] b)]
      (concat a b))))

(defonce _huff_tw_
  ;; converts :tw to :class
  (alter-var-root
    (var h/emit-attrs)
    (fn [f]
      (fn [append! attrs]
        (if (:tw attrs)
          (f append! (-> attrs
                         (update :class merge-attrs (:tw attrs))
                         (dissoc :tw)))
          (f append! attrs))))))

(defonce _huff_parse_opts_
  ;; renames attr keys
  (alter-var-root
    (var h/emit-attrs)
    (fn [f]
      (fn [append! attrs]
        (f append! (parse-opts attrs))))))

(def render h/html)

