(ns bloom.commons.pages
  ";; call at some point to initialize:

   (initialize! [{:id :home
                  :view some-view-fn
                  :path \"/\"}
                 {:id :profile
                  :view some-view-fn
                  :path \"/profile/:id\"
                  :coerce {:id integer?}}
                 ...])

   ;; include a reagent-view somewhere:
   [page-view]

   ;; to generate link string:
   (path-for :profile {:id 5})"
  (:require
    [reagent.core :as r]
    [reitit.core :as reitit]
    [reitit.coercion.spec]
    [reitit.coercion :as coercion]
    [accountant.core :as accountant]))

(defonce initialized? (atom false))
(defonce pages-by-id (atom {}))
(defonce router (atom nil))

(defonce state (r/atom {:page-id nil
                        :params nil}))

(defn- key-by [k arr]
  (reduce (fn [memo item]
            (assoc memo (k item) item)) {} arr))

(defn initialize!
  "Expects a list of pages, each a map with the following keys:
      :id         keyword, used in path-for
      :view       reagent view fn (recommend using #'view-fn)
                  receives params of page
      :path       string defining path
                  may include param patterns in path ex. /foo/:id
                  (params must also be included in :coerce)
      :coerce     map, data-spec coercion of path
      :on-enter!  fn to call when page is navigated to
                  (right after reagent state is updated with new page-id)
                  receives params (of new page)
      :on-exit!   fn to call when page is navigated away from
                  receives params (of old page)"
  [pages]
  (when-not @initialized?
    (reset! initialized? true)
    (reset! pages-by-id (key-by :id pages))

    (reset! router (reitit/router (->> pages
                                       (map (fn [page]
                                              (if (page :coerce)
                                                [(page :path) {:name (page :id)
                                                               :coercion reitit.coercion.spec/coercion
                                                               :parameters {:path (page :coerce)}}]
                                                [(page :path) (page :id)]))))
                                  {:compile coercion/compile-request-coercers}))

    (accountant/configure-navigation!
      {:nav-handler (fn [path]
                      (when-let [current-page-id (@state :page-id)]
                       (when-let [on-exit! (get-in @pages-by-id [current-page-id :on-exit!])]
                         (on-exit! (@state :params))))

                      (when-let [match (reitit/match-by-path @router path)]
                        (let [page-id (get-in match [:data :name])]
                          (reset! state {:page-id page-id
                                         :params (:path (coercion/coerce! match))})
                          (when-let [on-enter! (get-in @pages-by-id [page-id :on-enter!])]
                            (on-enter! (@state :params))))))
       :path-exists? (fn [path]
                       (boolean (reitit/match-by-path @router path)))})
    (accountant/dispatch-current!)))

(defn current-page-view []
  (let [{:keys [page-id params]} @state]
    (when-let [view (get-in @pages-by-id [page-id :view])]
      [view params])))

(defn path-for
  ([page-id]
   (path-for page-id {}))
  ([page-id params]
   (let [match (reitit/match-by-name @router page-id params)]
     (:path match))))

(defn navigate-to!
  ([page-id]
   (navigate-to! page-id {}))
  ([page-id params]
   (accountant/navigate! (path-for page-id params))))

;; TODO
;; include all page styles someplace
;; spec out register-pages!
