(ns prism.services
  (:require
    [prism.core :refer [defdelayed] :as prism]
    [prism.http :as http])
  (:import
    (java.net ConnectException)))

(defn- host-port-url [{:keys [Address ServicePort]}]
  {:host Address
   :port ServicePort
   :url  (format "http://%s:%s" Address ServicePort)})

(defdelayed ^:private consul-svc (:consul-url (prism/config)))

(defdelayed ^:private static-services {:consul [(consul-svc)]})

(defn- lookup-svc [{:keys [service tag]}]
  (some->> (http/request (cond-> {:url (str (:url (consul-svc)) "/v1/catalog/service/" (name service))
                                  :as  :json}
                                 tag (assoc :query-params {:filter (format "ServiceTags contains \"%s\"" (name tag))})))
           :body
           (mapv host-port-url)))

(defdelayed ^:private service-registry (:services (prism/config)))

(defdelayed ^:private services (atom (static-services)))

(defn- resolve-service! [service-kw]
  (when-let [resolved (-> (service-kw (service-registry))
                          lookup-svc
                          not-empty)]
    (-> (swap! (services) assoc service-kw resolved)
        service-kw)))

(defn- service! [service-kw]
  (or (service-kw @(services))
      (resolve-service! service-kw)))

(defn invalidate-service! [service-kw]
  (when-not (service-kw (static-services))
    (swap! (services) dissoc service-kw)))

(defn service-url! [svc]
  (-> (service! svc)
      rand-nth
      :url))

(defn- pathed-service-url! [service path]
  (some-> (service-url! service)
          (str path)))

(defn service-request [service path req]
  (if-let [url (pathed-service-url! service path)]
    (try
      (http/request (assoc req :url url))
      (catch ConnectException _
        (invalidate-service! service)
        (if-let [url (pathed-service-url! service path)]
          (http/request (assoc req :url url))
          (throw (ex-info "Could not resolve domain name for service" {:service  service
                                                                       :status   503
                                                                       :previous url})))))
    (throw (ex-info "Could not resolve domain name for service"
                    {:service service
                     :path    path
                     :status  503}))))
