(ns prism.cache
  (:require
    [prism.core :refer [defdelayed] :as prism]
    [cloffeine.cache :as c]
    [clojure.core.cache :as ccc])
  (:import
    (clojure.lang Associative Counted ILookup IPersistentCollection IPersistentMap Seqable)))

(defdelayed caches (-> (prism/config)
                       :cache
                       (update-vals c/make-cache)))

(deftype CloffeineCache [cache]
  ccc/CacheProtocol
  (lookup [_ k] (c/get-if-present cache k))
  (lookup [_ k not-found] (or (c/get-if-present cache k) not-found))
  (has? [_ e] (some? (c/get-if-present cache e)))
  (hit [this _] this)
  (miss [this k v] (c/put! cache k v) this)
  (evict [this e] (c/invalidate! cache e) this)
  (seed [this _] this)

  Associative
  (assoc [this k v] (ccc/miss this k v))
  (entryAt [_ k] (find (c/as-map cache) k))
  (containsKey [this k] (ccc/has? this k))

  ILookup
  (valAt [this k] (ccc/lookup this k))
  (valAt [this k v] (ccc/lookup this k v))

  Seqable
  (seq [_] (seq (c/as-map cache)))

  IPersistentCollection
  (count [_] (count (c/as-map cache)))
  (cons [_ _] (throw (UnsupportedOperationException.)))
  (empty [this] (c/invalidate-all! cache) this)
  (equiv [this other] (identical? this other))

  Counted
  IPersistentMap
  (without [this k] (ccc/evict this k) this)

  Iterable
  (iterator [_] (.iterator ^Iterable (sequence (c/as-map cache)))))

(defn put! [cachekw k v]
  (c/put! (cachekw (caches)) k v)
  v)

(defn get-if-present [cachekw k]
  (c/get-if-present (cachekw (caches)) k))

(defn invalidate! [cachekw k]
  (c/invalidate! (cachekw (caches)) k))

(defn get-or-put! [cachekw k f]
  (c/get (cachekw (caches)) k f))

