(ns prism.http
  (:require
    [clojure.java.io :as io]
    [hato.client :as hato]
    [hato.middleware :as hmw]
    [prism.core :refer [defdelayed]]
    [prism.internal.classpath :as cp]))

(defdelayed ^:private default-client (hato/build-http-client {:connect-timeout 2000}))

(def ^:private wrap-retry-request nil)
(declare ^:private attempt-request)
(cp/when-ns 'again.core
  (cp/when-ns 'taoensso.timbre
    (let [cb (fn [attempt-report]
               (when (#{:retry :attempt} (:again.core/status attempt-report))
                 (let [e (:again.core/exception attempt-report)
                       data (ex-data e)
                       request (-> data :request (select-keys [:url :method]))
                       response (-> data (select-keys [:status :body]))]
                   (taoensso.timbre/with-context
                     (merge request response)
                     (taoensso.timbre/warnf e "Error while making HTTP request."))))
               (when (-> (:again.core/exception attempt-report)
                         ex-data
                         :status
                         (= 400))
                 :again.core/fail))
          retry-config (fn [{request-retry :retry}]
                         {:again.core/callback cb
                          :again.core/strategy (cond
                                                 (vector? request-retry) request-retry
                                                 (number? request-retry) (take request-retry (list* 1000 2000 (repeat 3000)))
                                                 (true? request-retry) [1000 2000 3000 3000])})]

      (defn- wrap-retry-request [client]
        (fn
          ([req]
           (if (:retry req)
             (again.core/with-retries
               (retry-config req)
               (client req))
             (client req)))
          ([req respond raise]
           (if (:retry req)
             (client req
                     #(again.core/with-retries
                        (retry-config req)
                        (respond %))
                     raise)
             (client req respond raise))))))))

(defn- attempt-request [f req]
  (try
    (f req)
    (catch Exception e
      (cp/when-ns 'taoensso.timbre
        (taoensso.timbre/debugf "HTTP Exception: %s" e))
      e)))

(defn- wrap-catch-exceptions [client]
  (fn
    ([req]
     (if (:catch-exceptions? req)
       (attempt-request client req)
       (client req)))
    ([req respond raise]
     (if (:catch-exceptions? req)
       (client req #(attempt-request respond %) raise)
       (client req respond raise)))))

(def ^:private wrap-json-body nil)
(cp/when-ns 'prism.json
  (defmethod hmw/coerce-response-body :json [{:keys [coerce]} {:keys [body status] :as resp}]
    (let [^String charset (or (-> resp :content-type-params :charset) "UTF-8")]
      (cond
        (and (hmw/unexceptional-status? status)
             (or (nil? coerce) (= coerce :unexceptional)))
        (with-open [r (io/reader body :encoding charset)]
          (assoc resp :body (prism.json/json->clj r)))
        (= coerce :always)
        (with-open [r (io/reader body :encoding charset)]
          (assoc resp :body (prism.json/json->clj r)))
        (and (not (hmw/unexceptional-status? status)) (= coerce :exceptional))
        (with-open [r (io/reader body :encoding charset)]
          (assoc resp :body (prism.json/json->clj r)))
        :else resp)))

  (defn- wrap-json-body [client]
    (let [add-json-body (fn [req]
                          (if-some [body-json (:json req)]
                            (-> (assoc req :body (prism.json/write-json-string body-json)
                                           :content-type :json))
                            req))]
      (fn
        ([req]
         (-> (add-json-body req)
             client))
        ([req respond raise]
         (-> (add-json-body req)
             (client respond raise)))))))

(def ^:private wrapped-request (->> [wrap-json-body
                                     wrap-retry-request
                                     wrap-catch-exceptions]
                                    (into hmw/default-middleware (filter some?))
                                    (hmw/wrap-request hato/request*)))

(defn request [req]
  (let [req (-> (update req :http-client #(or % (default-client)))
                (update :timeout (fnil identity 10000)))]
    (if (:async? req)
      (wrapped-request req identity #(throw %))
      (wrapped-request req))))

(comment
  (request {:url               "https://webhook.site/92f0a106-6e73-4cec-a5cc-18d0298bab51"
            :method            :post
            ;:throw-exceptions false ;true by default
            :json              {:abc 123}
            :retry             [1]
            :catch-exceptions? true}))
