(ns prism.http-server
  (:require
    [muuntaja.core :as m]
    [muuntaja.interceptor :as muuntaja]
    [prism.core :as prism]
    [prism.internal.classpath :as cp]
    [reitit.coercion.malli :as mc]
    [reitit.core :as r]
    [reitit.dev.pretty :as pretty]
    [reitit.http :as rhttp]
    [reitit.http.coercion :as rhc]
    [reitit.http.interceptors.exception :as exception]
    [reitit.http.interceptors.parameters :as parameters]
    [reitit.interceptor.sieppari :as sieppari]
    [reitit.ring :as ring]
    [reitit.spec :as rs]
    [ring.adapter.jetty9 :as jetty])
  (:import
    (org.eclipse.jetty.server Server)))

(def muuntaja-instance (cp/if-ns 'prism.json
                         (-> (update m/default-options :formats assoc "application/json" prism.json/muuntaja-format)
                             m/create)
                         m/instance))

(defn- reference-expand [registry]
  (fn [data opts]
    (if (keyword? data)
      (some-> data
              registry
              (r/expand opts)
              (assoc :name data))
      (r/expand data opts))))

(defn- exception-handler [message exception request]
  {:status (-> (ex-data exception)
               (:status 500))
   :body   {:message   message
            :exception (str exception)
            :uri       (:uri request)}})

(defn- exception-interceptor []
  (exception/exception-interceptor
    (merge
      exception/default-handlers
      {::error             (partial exception-handler "error")
       ::exception         (partial exception-handler "exception")
       ::exception/default (partial exception-handler "default")
       ::exception/wrap    (fn [handler e request]
                             (cp/when-ns 'taoensso.timbre
                               (when-not (-> e ex-data :status)
                                 (taoensso.timbre/with-context
                                   (select-keys request [:request-method :scheme :content-length :content-type :uri
                                                         :parameters :query-params :path-params :query-string :body-params])
                                   (taoensso.timbre/errorf
                                     e "Unhandled error on request"))))
                             (handler e request))})))

(defn default-interceptors [muuntaja]
  [(rhc/coerce-response-interceptor)
   (muuntaja/format-interceptor muuntaja)
   (parameters/parameters-interceptor)
   (exception-interceptor)
   (rhc/coerce-request-interceptor)])

(defn create-router [routes-data & {:keys [expand expand-map coercion muuntaja interceptors
                                           inject-match? inject-router?]
                                    :or   {coercion mc/coercion
                                           muuntaja muuntaja-instance
                                           inject-match? false
                                           inject-router? false}}]
  (let [expand (or expand
                   (when (seq expand-map)
                     (reference-expand expand-map))
                   r/expand)
        interceptors (or interceptors
                         (default-interceptors muuntaja))]
    (rhttp/ring-handler
      (rhttp/router
        routes-data
        {:expand    expand
         :exception pretty/exception
         :validate  rs/validate
         :data      {:coercion coercion}})

      (ring/routes
        (ring/create-default-handler))

      {:executor       sieppari/executor
       :interceptors   interceptors
       :inject-match?  inject-match?
       :inject-router? inject-router?})))

(defn- configure-server [^Server server]
  (.addShutdownHook (Runtime/getRuntime)
                    (Thread. #(do
                                (cp/when-ns 'taoensso.timbre
                                  (taoensso.timbre/info "Stopping server"))
                                (.stop server)
                                (cp/when-ns 'prism.postgres
                                  (cp/when-ns 'taoensso.timbre
                                    (taoensso.timbre/info "Shutting down DB connection pool"))
                                  (.close (prism.postgres/data-src)))))))

(defn start-router! ^Server [router]
  (let [port (-> (prism/config) :port)]
    (cp/when-ns 'taoensso.timbre
      (taoensso.timbre/infof "Starting server on port %d" port))
    (jetty/run-jetty router {:port         port
                             :h2c?         true
                             :join?        false
                             :configurator configure-server})))

(comment
  (require '[reitit.openapi :as openapi])
  (require '[prism.http :as http])
  (require '[clojure.java.io :as io])
  (->> (create-router
         [["/openapi.json"
           {:get {:handler        (openapi/create-openapi-handler)
                  :openapi        {:info {:title "my nice api" :version "0.0.1"}}
                  :inject-router? true
                  :no-doc         true}}]
          ["/test/:abc"
           ["" {:get {:handler (fn [req]
                                 (pr req)
                                 (println)
                                 {:status 200})}}]
           ["/body" {:get {:handler (fn [req]
                                      (pr req)
                                      (println)
                                      {:status 400
                                       :body   {:with "body"}})}}]
           ["/error" {:put {:handler    (fn [req]
                                          [req]
                                          (pr req)
                                          (println)
                                          (throw (ex-info "short and stout" {:status 418})))
                            :parameters {:body [:map
                                                [:jay [:enum "son" "daughter"]]]}}}]]]
         :interceptors (conj (default-interceptors muuntaja-instance) openapi/openapi-feature))
       start-router!
       (def server))
  (.stop server)
  (->> (http/request {:method :get
                      :url    (str "http://localhost:" (-> (prism/config) :port) "/openapi.json")})
       :body
       (spit (io/file "dev/openapi.json")))
  (http/request {:method :get
                 :url    (str "http://localhost:" (-> (prism/config) :port) "/test/value")})
  (http/request {:method           :get
                 :throw-exceptions false
                 :as               :auto
                 :url              (str "http://localhost:" (-> (prism/config) :port) "/test/value/body")})
  (http/request {:method           :put
                 :timeout          100000
                 :throw-exceptions false
                 :json             {:jay "x"}
                 :url              (str "http://localhost:" (-> (prism/config) :port) "/test/value/error")}))
