(ns prism.http-server
  (:require
    [malli.core :as m]
    [malli.transform :as mt]
    [malli.util :as mu]
    [muuntaja.core :as muuntaja]
    [muuntaja.interceptor :as mi]
    [prism.core :as prism]
    [prism.internal.classpath :as cp]
    [reitit.coercion.malli :as mc]
    [reitit.core :as r]
    [reitit.http :as rhttp]
    [reitit.http.coercion :as rhc]
    [reitit.http.interceptors.exception :as exception]
    [reitit.http.interceptors.parameters :as parameters]
    [reitit.interceptor.sieppari :as sieppari]
    [reitit.ring :as ring]
    [reitit.spec :as rs]
    [ring.adapter.jetty9 :as jetty])
  (:import
    (java.time Instant)
    (org.eclipse.jetty.server Server)))

(def muuntaja-instance (cp/if-ns 'prism.json
                         (-> (update muuntaja/default-options :formats assoc "application/json" prism.json/muuntaja-format)
                             muuntaja/create)
                         muuntaja/instance))

(defn- reference-expand [registry]
  (fn [data opts]
    (if (keyword? data)
      (some-> data
              registry
              (r/expand opts)
              (assoc :name data))
      (r/expand data opts))))

(defn- exception-handler [message exception request]
  {:status (-> (ex-data exception)
               (:status 500))
   :body   {:message   message
            :exception (str exception)
            :uri       (:uri request)}})

(defn- exception-interceptor []
  (exception/exception-interceptor
    (merge
      exception/default-handlers
      {::error             (partial exception-handler "error")
       ::exception         (partial exception-handler "exception")
       ::exception/default (partial exception-handler "default")
       ::exception/wrap    (fn [handler e request]
                             (cp/when-ns 'taoensso.timbre
                               (when-not (-> e ex-data :status)
                                 (taoensso.timbre/with-context
                                   (select-keys request [:request-method :scheme :content-length :content-type :uri
                                                         :parameters :query-params :path-params :query-string :body-params])
                                   (taoensso.timbre/errorf
                                     e "Unhandled error on request"))))
                             (handler e request))})))

(defn default-interceptors [muuntaja]
  [(rhc/coerce-response-interceptor)
   (mi/format-interceptor muuntaja)
   (parameters/parameters-interceptor)
   (exception-interceptor)
   (rhc/coerce-request-interceptor)])

(defn -json-transformer [strip-extra-keys?]
  (mt/transformer
    {:name     :inst
     :decoders {'inst? #(if (string? %) (Instant/parse ^String %) %)}
     :encoders {'inst? mt/-any->string}}
    (when strip-extra-keys?
      (-> (mt/strip-extra-keys-transformer)
          m/-transformer-chain
          first
          (update-vals #(dissoc % :map-of))))
    mt/default-value-transformer
    mt/json-transformer))

(def string-transformer
  (mt/transformer
    {:name     :inst
     :decoders {'inst? #(if (string? %) (Instant/parse ^String %) %)}
     :encoders {'inst? mt/-any->string}}
    mt/string-transformer
    mt/default-value-transformer))

(def default-coerce-options
  (let [json-transformer (-json-transformer false)]
    {:transformers {:body     {:default json-transformer
                               :formats {"application/json" json-transformer}}
                    :string   {:default string-transformer}
                    :response {:default json-transformer}}
     :compile      mu/open-schema}))

(defn create-router [routes-data & {:keys [expand expand-map coerce-options muuntaja interceptors
                                           inject-match? inject-router?]
                                    :or   {muuntaja       muuntaja-instance
                                           inject-match?  false
                                           inject-router? false}}]
  (let [expand (or expand
                   (when (seq expand-map)
                     (reference-expand expand-map))
                   r/expand)
        interceptors (or interceptors
                         (default-interceptors muuntaja))
        coercion (-> (merge default-coerce-options coerce-options)
                     mc/create)]
    (rhttp/ring-handler
      (rhttp/router
        routes-data
        {:expand    expand
         :validate  rs/validate
         :data      {:coercion coercion}})

      (ring/routes
        (ring/create-default-handler))

      {:executor       sieppari/executor
       :interceptors   interceptors
       :inject-match?  inject-match?
       :inject-router? inject-router?})))

(defn- configure-server [^Server server]
  (.addShutdownHook (Runtime/getRuntime)
                    (Thread. #(do
                                (cp/when-ns 'taoensso.timbre
                                  (taoensso.timbre/info "Stopping server"))
                                (.stop server)
                                (cp/when-ns 'prism.postgres
                                  (cp/when-ns 'taoensso.timbre
                                    (taoensso.timbre/info "Shutting down DB connection pool"))
                                  (.close (prism.postgres/data-src)))))))

(defn start-router! ^Server [router]
  (let [port (-> (prism/config) :port)]
    (cp/when-ns 'taoensso.timbre
      (taoensso.timbre/infof "Starting server on port %d" port))
    (jetty/run-jetty router {:port         port
                             :h2c?         true
                             :join?        false
                             :configurator configure-server})))

(comment
  (require '[reitit.openapi :as openapi])
  (require '[prism.http :as http])
  (require '[clojure.java.io :as io])
  (->> (create-router
         [["/openapi.json"
           {:get {:handler        (openapi/create-openapi-handler)
                  :openapi        {:info {:title "my nice api" :version "0.0.1"}}
                  :inject-router? true
                  :no-doc         true}}]
          ["/test/:abc"
           ["" {:get {:handler (fn [req]
                                 (pr req)
                                 (println)
                                 {:status 200})}}]
           ["/body" {:get {:handler    (fn [req]
                                         (pr req)
                                         (println)
                                         {:status 201
                                          :body   {:with "body"}})
                           :parameters {:body [:map [:a int?]]
                                        :path map?}}}]
           ["/error" {:put {:handler    (fn [req]
                                          [req]
                                          (pr req)
                                          (println)
                                          (throw (ex-info "short and stout" {:status 418})))
                            :parameters {:body [:map
                                                [:jay [:enum "son" "daughter"]]]}}}]]]
         :interceptors (conj (default-interceptors muuntaja-instance) openapi/openapi-feature))
       start-router!
       (def server))
  (.stop server)
  (->> (http/request {:method :get
                      :url    (str "http://localhost:" (-> (prism/config) :port) "/openapi.json")})
       :body
       (spit (io/file "dev/openapi.json")))
  (http/request {:method :get
                 :url    (str "http://localhost:" (-> (prism/config) :port) "/test/value")})
  (http/request {:method           :get
                 :throw-exceptions false
                 :as               :auto
                 :json             {:a 1 :b 2}
                 :url              (str "http://localhost:" (-> (prism/config) :port) "/test/123/body")})
  (http/request {:method           :put
                 :timeout          100000
                 :throw-exceptions false
                 :json             {:jay "x"}
                 :url              (str "http://localhost:" (-> (prism/config) :port) "/test/value/error")}))
