(ns prism.redis
  (:require
    [prism.core :as prism :refer [defdelayed]]
    [taoensso.carmine :as car]
    [taoensso.timbre :as log]))

(defn- connection-pool [cfg]
  {:pool (car/connection-pool {})
   :spec cfg})

(defdelayed conns (-> (prism/config)
                      :redis
                      (update-vals connection-pool)))

(defmacro wcar* [conn-name & body]
  `(let [c# (~conn-name (conns))]
     (try
       (car/wcar c# ~@body)
       (catch Exception e#
         (log/error e#)))))

(defn hexpire-at [k expire-at-secs members]
  (car/redis-call (into ["hexpireat" k expire-at-secs "FIELDS" (str (count members))]
                        members)))

(defn hset [k score-members]
  (when (seq score-members)
    (apply car/hset k score-members)))

(defn lpush [k members]
  (when (seq members)
    (apply car/lpush k members)))

(defn set-kv!* [c {:keys [k v expire get? exclusive]}]
  (when v
    (let [resp (wcar*
                 c
                 (apply car/set
                        (cond-> [k v]
                                (#{:nx :xx} exclusive) (conj exclusive)
                                get? (conj :get)
                                (integer? expire) (conj :ex expire))))]
      (if get?
        resp
        v))))

(defn get-val* [c k]
  (wcar* c (car/get k)))

(defn get-or-set!* [c {:keys [k f] :as entry}]
  (or (get-val* c k)
      (some->> (f)
               (assoc entry :get? true :v)
               (set-kv!* c))))

(defn hset!* [c {:keys [k member v]}]
  (when v
    (wcar* c (car/hset k member v))
    v))

(defn hdel!* [c {:keys [k member]}]
  (wcar* c (car/hdel k member)))

(defn del!* [c k]
  (wcar* c (car/del k)))

(defn hvals* [c k]
  (wcar* c (car/hvals k)))

(defn lrange-all* [c k]
  (wcar* c (car/lrange k 0 -1)))

(comment
  (get-val* :local :test2)
  (wcar* :local (car/get :test))
  (get-or-set!*
    :local
    {:k :test2
     :f (fn [] "abc")}))
