(ns prism.ilp
  (:require
    [honey.sql :as hsql])
  (:import
    (java.math RoundingMode)
    (java.time Instant)))

;https://github.com/influxdata/influxdb-client-java/blob/master/client/src/main/java/com/influxdb/client/write/Point.java

(def ^:private nanos-per-second (BigInteger/valueOf (* 1000 1000 1000)))

(defn- ->sorted [coll]
  (if (sorted? coll)
    coll
    (sort coll)))

(defn- append-escaped-measurement! ^StringBuilder [^StringBuilder sb measurement]
  (doseq [c (hsql/format-entity measurement)]
    (case c
      \newline (.append sb "\\n")
      \return (.append sb "\\r")
      \tab (.append sb "\\t")
      (\space \,) (-> (.append sb \\)
                      (.append ^char c))
      (.append sb ^char c)))
  sb)

(defn- append-escaped-key! ^StringBuilder [^StringBuilder sb k]
  (doseq [c (hsql/format-entity k)]
    (case c
      \newline (.append sb "\\n")
      \return (.append sb "\\r")
      \tab (.append sb "\\t")
      (\space \, \=) (-> (.append sb \\)
                         (.append ^char c))
      (.append sb ^char c)))
  sb)

(defn- append-escaped-value! ^StringBuilder [^StringBuilder sb s]
  (doseq [c s]
    (-> (case c
          (\\ \") (.append sb \\)
          sb)
        (.append ^char c)))
  sb)

(defn- append-tags! ^StringBuilder [^StringBuilder sb tags]
  (-> ^StringBuilder (reduce-kv
                       (fn [^StringBuilder sb k v]
                         (-> (.append sb \,)
                             (append-escaped-key! k)
                             (.append \=)
                             (append-escaped-key! v)))
                       sb
                       (->sorted tags))
      (.append \space)))

(defn- instant->nanos [^Instant inst]
  (-> (BigInteger/valueOf (.getEpochSecond inst))
      (.multiply nanos-per-second)
      (.add (BigInteger/valueOf (.getNano inst)))))

(defn- append-field-value! ^StringBuilder [^StringBuilder sb field-metadata v]
  (cond
    (or (float? v) (decimal? v)) (let [bd (BigDecimal. ^double v)
                                       decimal-places (-> (.precision bd)
                                                          (min 340))
                                       bd (.setScale bd decimal-places RoundingMode/HALF_UP)]
                                   (.append sb bd))
    (int? v) (-> (.append sb ^long v)
                 (.append ^char (if (::unsigned? field-metadata)
                                  \u
                                  \i)))
    (number? v) (-> (.append sb (str v))
                    (.append ^char (if (::unsigned? field-metadata)
                                     \u
                                     \i)))
    (instance? Instant v) (recur sb (assoc field-metadata ::unsigned? true) (instant->nanos v))
    :else (-> (.append sb \")
              (append-escaped-value! (str v))
              (.append \"))))

(defn- dec-length! [^StringBuilder sb]
  (doto sb
    (.setLength (max (dec (.length sb))
                     0))))

(defn- append-fields! [^StringBuilder sb fields]
  (let [^StringBuilder sb (reduce-kv
                            (fn [^StringBuilder sb k v]
                              (-> (append-escaped-key! sb k)
                                  (.append \=)
                                  (append-field-value! (-> (meta fields)
                                                           (get k)) v)
                                  (.append \,)))
                            sb
                            (->sorted fields))]
    (dec-length! sb)))

(defn- append-time! [^StringBuilder sb ^Instant time]
  (if time
    (let [time (instant->nanos time)]
      (-> (.append sb \space)
          (.append time)))
    sb))

(defn- append-point! ^StringBuilder [^StringBuilder sb point]
  (-> (append-escaped-measurement! sb (:measurement point))
      (append-tags! (:tags point))
      (append-fields! (:fields point))
      (append-time! (:time point))))

(defn points->string [points]
  (-> (reduce (fn [sb point]
                (-> (append-point! sb point)
                    (.append \newline))) (StringBuilder.) points)
      dec-length!
      str
      not-empty))

(comment
  (points->string [{:measurement "measurement"
                    :tags        {"x" "has \" quote"
                                  "y" "1"}
                    :fields      {"x" "has space"
                                  "y" 2.0123456789012345678}
                    :time        (Instant/now)}]))
