(ns prism.s3
  (:require
    [clojure.java.io :as io]
    [hato.middleware :as hmw]
    [prism.core :refer [defdelayed] :as prism]
    [prism.internal.classpath :as cp]
    [prism.services :as services])
  (:import
    (com.github.davidmoten.aws.lw.client BaseUrlFactory Client HttpMethod Multipart Request)
    (com.github.davidmoten.aws.lw.client.xml.builder Xml)
    (java.net ConnectException)
    (java.nio.charset StandardCharsets)
    (java.security MessageDigest)
    (java.util Base64 Map)
    (org.json XML)))

(def ^:private dns-url-factory
  (reify
    BaseUrlFactory
    (create [_ _ _]
      (services/service-url! :s3))))

(defdelayed ^:private ^Client s3-client
  (let [{{:keys [username password]} :s3} (prism/config)]
    (-> (Client/s3)
        (.region "us-east-1")
        (.accessKey username)
        (.secretKey password)
        (.baseUrlFactory dns-url-factory)
        .build)))

(defn- parse-response-stream [input-stream]
  (->> input-stream io/reader XML/toJSONObject .toMap))

(defn- from-connect-exception? [e]
  (loop [e e]
    (cond
      (instance? ConnectException e) true
      (some? e) (recur (ex-cause e))
      :else false)))

(defn- with-connection-handling [f]
  (try
    (f)
    (catch Exception e
      (if (from-connect-exception? e)
        (do (services/invalidate-service! :s3)
            (when-not (services/service-url! :s3)
              (throw (ex-info "Could not resolve domain name for s3" {:status 503} e))))
        (throw e)))))

(defn- make-request! [^Request request]
  (if-let [response (with-connection-handling #(.responseInputStream request))]
    (parse-response-stream response)
    (recur request)))

(defn list-objects [{:keys [bucket delimiter prefix]
                     :or   {delimiter "/"}}]
  (-> (.path (s3-client) (into-array String [bucket]))
      (.query "list-type" "2")
      (.query "delimiter" delimiter)
      (.query "prefix" prefix)
      make-request!
      (get "ListBucketResult")))

(defn delete-object! [{:keys [bucket key]}]
  (-> (.path (s3-client) (into-array String [bucket key]))
      (.method HttpMethod/DELETE)
      make-request!))

(defn put-object! [{:keys [bucket key body]}]
  (-> (.path (s3-client) (into-array String [bucket key]))
      (.requestBody ^String body)
      (.method HttpMethod/PUT)
      make-request!))

(defn upload-object! [{:keys [bucket key body]}]
  (let [req (-> (Multipart/s3 (s3-client))
                (.bucket bucket)
                (.key key))]
    (with-connection-handling
      #(.upload req ^Callable (constantly body)))))

(defn- stream-response [^Request req]
  (let [stream (.responseInputStream req)]
    (if (-> (.statusCode stream)
            hmw/unexceptional-status?)
      stream
      (cp/when-ns 'taoensso.timbre
        (let [error (-> (parse-response-stream stream)
                        (get "Error"))]
          (when-not (->> (get error "Code")
                         (= "NoSuchKey"))
            (taoensso.timbre/warnf "Received errored s3 response: '%s'" error)))))))

(defn get-object [{:keys [bucket key]}]
  (let [req (.path (s3-client) (into-array String [bucket key]))]
    (with-connection-handling
      #(stream-response req))))

(defn get-object-metadata [{:keys [bucket key]}]
  (with-connection-handling
    #(-> (.path (s3-client) (into-array String [bucket key]))
         (.method HttpMethod/HEAD)
         .response
         .headers)))

(defn- delete-objects-request ^String [ks]
  (-> (reduce
        (fn [^Xml xml k]
          (-> (.element xml "Object")
              (.element "Key")
              (.content k)
              .up
              .up))
        (-> (Xml/create "Delete")
            (.attribute "xmlns" "http://s3.amazonaws.com/doc/2006-03-01/"))
        ks)
      str))

(defn- aws-md5-header [body-bytes]
  (let [md (MessageDigest/getInstance "MD5")]
    (.update md ^bytes body-bytes)
    (-> (.encode (Base64/getEncoder) (.digest md))
        (String. StandardCharsets/US_ASCII))))

(defn delete-objects! [{:keys [bucket] :as opts}]
  (when-let [keys (as-> (list-objects opts) $
                        (get $ "Contents")
                        (if (instance? Map $) (vector $) $)
                        (mapv #(get % "Key") $)
                        (seq $))]
    (let [body (-> (delete-objects-request keys)
                   .getBytes)]
      (-> (.path (s3-client) (into-array String [bucket "?delete"]))
          (.requestBody ^bytes body)
          (.header "Content-MD5" (aws-md5-header body))
          (.method HttpMethod/POST)
          make-request!))))

(comment
  (get-object {:bucket "ffwd-temp-test"
               :key    "some-key"})
  (put-object! {:bucket "ffwd-temp-test"
                :key    "some-key"
                :body   (slurp (io/file "deps.edn"))})
  (list-objects {:prefix "rules/"
                 :bucket "ffwd-loki"})
  (delete-objects! {:prefix    (str "pythia/log-anomaly/" "f6295714-d7f3-4eea-a626-5dddce8a1caf" "/")
                    :bucket    "ffwd-temp-test"
                    :delimiter ""}))

